package com.atlassian.crowd.directory.query;

import com.atlassian.crowd.embedded.api.SearchRestriction;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.entity.restriction.BooleanRestriction;
import com.atlassian.crowd.search.query.entity.restriction.MatchMode;
import com.atlassian.crowd.search.query.entity.restriction.Property;
import com.atlassian.crowd.search.query.entity.restriction.PropertyRestriction;
import com.atlassian.crowd.search.query.entity.restriction.constants.GroupTermKeys;
import com.atlassian.crowd.search.query.entity.restriction.constants.UserTermKeys;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static java.util.stream.Collectors.joining;

/**
 * Translates Crowd queries into filters used by Microsoft Graph.
 */
public class MicrosoftGraphQueryTranslator {

    public static final String USERNAME = "userPrincipalName";
    public static final String FIRST_NAME = "givenName";
    public static final String LAST_NAME = "surname";
    public static final String DISPLAY_NAME = "displayName";
    public static final String MAIL = "mail";
    public static final String ID = "id";
    public static final String ACCOUNT_ENABLED = "accountEnabled";
    static final ODataSelect ID_SELECT = new ODataSelect(ID);
    static final ODataSelect USERNAME_SELECT = new ODataSelect(USERNAME);
    static final ODataSelect MINIMAL_USER_SELECT = new ODataSelect(USERNAME, ID);
    public static final ODataSelect USER_SELECT = new ODataSelect(USERNAME, FIRST_NAME, LAST_NAME, DISPLAY_NAME, MAIL, ID, ACCOUNT_ENABLED);

    static final String GROUPNAME = "displayName";
    static final String DESCRIPTION = "description";
    static final String MEMBERS = "members";
    public static final ODataExpand MEMBERS_EXPAND = new ODataExpand(MEMBERS);
    public static final ODataSelect GROUPNAME_SELECT = new ODataSelect(GROUPNAME);
    static final ODataSelect MINIMAL_GROUP_SELECT = new ODataSelect(GROUPNAME, ID);
    public static final ODataSelect GROUP_SELECT = new ODataSelect(GROUPNAME, DESCRIPTION, ID);
    public static final ODataSelect DELTA_QUERY_GROUP_SELECT = GROUP_SELECT.addColumns(MEMBERS);

    static final ODataSelect NAMES_SELECT = new ODataSelect(USERNAME, GROUPNAME);

    static final ODataSelect USER_AND_GROUP_SELECT = USER_SELECT.merge(GROUP_SELECT);
    static final ODataSelect MINIMAL_USER_AND_GROUP_SELECT = MINIMAL_USER_SELECT.merge(MINIMAL_GROUP_SELECT);

    private static final String GRAPH_EQUALS_OPERATOR = "eq";
    private static final String GRAPH_LESS_THAN_OPERATOR = "lt";
    private static final String GRAPH_LESS_THAN_OR_EQUAL_OPERATOR = "lte";
    private static final String GRAPH_GREATER_THAN_OPERATOR = "gt";
    private static final String GRAPH_GREATER_THAN_OR_EQUAL_OPERATOR = "gte";
    private static final String GRAPH_STARTS_WITH_FUNCTION = "startswith";

    private static final Logger log = LoggerFactory.getLogger(MicrosoftGraphQueryTranslator.class);


    /**
     * Converts a Crowd EntityQuery to a GraphQuery that should be used to fetch the first page of results. This will
     * map the search restrictions, the columns to fetch, the start index and the amount of results to fetch
     *
     * @param query the Crowd query to convert
     * @param usernameAttribute name of the attribute that username restriction should be matched to
     * @return a GraphQuery for fetching the first page of results
     */
    public GraphQuery convert(final EntityQuery query, String usernameAttribute) {
        final SearchRestriction searchRestriction = query.getSearchRestriction();
        final ODataFilter oDataFilter = translateSearchRestriction(query.getEntityDescriptor(), searchRestriction, usernameAttribute);
        final ODataSelect oDataSelect = resolveAzureAdColumnsForSingleEntityTypeQuery(query.getEntityDescriptor(), query.getReturnType());
        ODataTop limit = resolveQueryLimit(query.getMaxResults());
        return new GraphQuery(oDataFilter, oDataSelect, query.getStartIndex(), limit);
    }

    private ODataTop resolveQueryLimit(final int maxResults) {
        return maxResults == EntityQuery.ALL_RESULTS ? ODataTop.FULL_PAGE : ODataTop.forSize(maxResults);
    }

    /**
     * @param entity     the entity for which we're querying
     * @param returnType the return type of the  Crowd query
     * @return an ODataSelect with the columns appropriate for the given entity and return type
     */
    public ODataSelect resolveAzureAdColumnsForSingleEntityTypeQuery(final EntityDescriptor entity, final Class<?> returnType) {
        return resolveAzureAdColumnsForSingleEntityTypeQuery(
                entity, returnType == String.class ? FetchMode.NAME : FetchMode.FULL);
    }

    /**
     * Converts a search restriction to an ODataFilter, which can be used in a graph query
     *
     * @param entityDescriptor the entityDescriptor of the entity that will be queried for
     * @param restriction      the restriction to translate
     * @param usernameAttribute name of the attribute that username restriction should be matched to
     * @return the translated filter
     */
    public ODataFilter translateSearchRestriction(final EntityDescriptor entityDescriptor, final SearchRestriction restriction, String usernameAttribute) {
        if (restriction instanceof PropertyRestriction) {
            return new ODataFilter(translatePropertyRestriction(entityDescriptor, (PropertyRestriction) restriction, usernameAttribute));
        } else if (restriction instanceof BooleanRestriction) {
            return new ODataFilter(translateBooleanRestriction(entityDescriptor, (BooleanRestriction) restriction, usernameAttribute));
        } else {
            return ODataFilter.EMPTY;
        }
    }

    /**
     * @param entity    the entity for which we're querying
     * @param fetchMode the types of columns that should be fetched
     * @return an ODataSelect with the columns appropriate for the given entity and fetch mode
     */
    @SuppressWarnings("Duplicates")
    public ODataSelect resolveAzureAdColumnsForSingleEntityTypeQuery(final EntityDescriptor entity, final FetchMode fetchMode) {
        if (entity == EntityDescriptor.user()) {
            switch (fetchMode) {
                case FULL:
                case DELTA_QUERY:
                    return USER_SELECT;
                case NAME_AND_ID:
                    return MINIMAL_USER_SELECT;
                case NAME:
                    return USERNAME_SELECT;
                case ID:
                    return ID_SELECT;
                default:
                    throw new IllegalArgumentException(String.format("Cannot translate query for entity %s, fetch mode %s", entity, fetchMode));
            }
        } else if (entity == EntityDescriptor.group()) {
            switch (fetchMode) {
                case FULL:
                    return GROUP_SELECT;
                case DELTA_QUERY:
                    return DELTA_QUERY_GROUP_SELECT;
                case NAME_AND_ID:
                    return MINIMAL_GROUP_SELECT;
                case NAME:
                    return GROUPNAME_SELECT;
                case ID:
                    return ID_SELECT;
                default:
                    throw new IllegalArgumentException(String.format("Cannot translate query for entity %s, fetch mode %s", entity, fetchMode));
            }
        } else {
            throw new IllegalArgumentException(String.format("Cannot translate query for entity %s, fetch mode %s", entity, fetchMode));
        }
    }

    /**
     * @param entity    the entity for which we're querying
     * @param fetchMode the types of navigation properties that should be fetched
     * @return an ODataExpand with the navigation properties appropriate for the given entity and fetch mode
     */
    public ODataExpand resolveAzureAdNavigationPropertiesForSingleEntityTypeQuery(final EntityDescriptor entity, final FetchMode fetchMode) {
        if (entity == EntityDescriptor.group()) {
            switch (fetchMode) {
                case DELTA_QUERY:
                    return MEMBERS_EXPAND;
                default:
                    throw new IllegalArgumentException(String.format("Cannot translate query for entity %s, fetch mode %s", entity, fetchMode));
            }
        } else {
            throw new IllegalArgumentException(String.format("Cannot translate query for entity %s, fetch mode %s", entity, fetchMode));
        }
    }

    /**
     * @param fetchMode the types of columns that should be fetched
     * @return an ODataSelect with the columns appropriate for a query that will fetch both users and groups and the
     *         specified fetch mode
     */
    public ODataSelect translateColumnsForUsersAndGroupsQuery(final FetchMode fetchMode) {
        switch (fetchMode) {
            case FULL:
                return USER_AND_GROUP_SELECT;
            case NAME_AND_ID:
                return MINIMAL_USER_AND_GROUP_SELECT;
            case NAME:
                return NAMES_SELECT;
            case ID:
                return ID_SELECT;
            default:
                throw new IllegalArgumentException(String.format("Cannot translate query for users and groups query, fetch mode %s", fetchMode));
        }
    }

    private String translateBooleanRestriction(final EntityDescriptor entityDescriptor, final BooleanRestriction restriction, String usernameAttribute) {
        String result = restriction.getRestrictions()
                .stream()
                .map(subrestriction -> translateSearchRestriction(entityDescriptor, subrestriction, usernameAttribute).asRawValue())
                .collect(joining(" " + restriction.getBooleanLogic().name() + " "));
        return restriction.getRestrictions().size() > 1 ? "(" + result + ")" : result;
    }

    private String translatePropertyRestriction(final EntityDescriptor entityDescriptor, final PropertyRestriction restriction, String usernameAttribute) {
        final StringBuilder stringBuilder = new StringBuilder();
        final String azureAdPropertyName = resolvePropertyName(entityDescriptor, restriction.getProperty(), usernameAttribute);
        if (restriction.getMatchMode() == MatchMode.CONTAINS
                || restriction.getMatchMode() == MatchMode.STARTS_WITH
                || restriction.getMatchMode() == MatchMode.ENDS_WITH) {
            appendODataFunction(restriction, stringBuilder, azureAdPropertyName);
        } else {
            appendODataComparison(restriction, stringBuilder, azureAdPropertyName);
        }
        return stringBuilder.toString();
    }

    private void appendODataComparison(final PropertyRestriction restriction, final StringBuilder stringBuilder, final String azureAdPropertyName) {
        stringBuilder.append(azureAdPropertyName)
                .append(" ")
                .append(resolveOperator(restriction.getMatchMode()))
                .append(" ");
        if (shouldBeQuoted(restriction)) {
            stringBuilder.append("'");
        }
        // MS graph uses an equals null comparison instead of is null
        if (restriction.getMatchMode() == MatchMode.NULL) {
            stringBuilder.append("null");
        } else {
            stringBuilder.append(sanitizeOdataValue(restriction.getValue()));
        }
        if (shouldBeQuoted(restriction)) {
            stringBuilder.append("'");
        }
    }

    private boolean shouldBeQuoted(final PropertyRestriction restriction) {
        return restriction.getProperty().getPropertyType() == String.class
                && restriction.getMatchMode() != MatchMode.NULL;
    }

    private void appendODataFunction(final PropertyRestriction restriction, final StringBuilder stringBuilder, final String azureAdPropertyName) {
        stringBuilder
                .append(resolveOperatorFunction(restriction.getMatchMode()))
                .append("(")
                .append(sanitizeOdataValue(azureAdPropertyName))
                .append(",");

        final boolean isStringRestriction = restriction.getProperty().getPropertyType() == String.class;
        if (isStringRestriction) {
            stringBuilder.append("'");
        }
        stringBuilder.append(sanitizeOdataValue(restriction.getValue()));
        if (isStringRestriction) {
            stringBuilder.append("'");
        }
        stringBuilder.append(")");
    }

    private String resolveOperatorFunction(final MatchMode matchMode) {
        switch (matchMode) {
            //FIXME: contains is borken for users, remove it when dir is syncable
            case CONTAINS:
                log.warn("Contains query is not supported for Azure AD directories, using 'starts with' instead");
                return GRAPH_STARTS_WITH_FUNCTION;
            case STARTS_WITH:
                return GRAPH_STARTS_WITH_FUNCTION;
            case ENDS_WITH:
                log.warn("'Ends with' query is not supported for Azure AD directories, using 'starts with' instead");
                return GRAPH_STARTS_WITH_FUNCTION;
            default:
                throw new IllegalArgumentException("Cannot query by match mode " + matchMode);
        }
    }

    private String resolveOperator(final MatchMode matchMode) {
        switch (matchMode) {
            case EXACTLY_MATCHES:
            case NULL: // MS graph supports a regular equals null comparison instead of an is null style
                return GRAPH_EQUALS_OPERATOR;
            case GREATER_THAN:
                return GRAPH_GREATER_THAN_OPERATOR;
            case GREATER_THAN_OR_EQUAL:
                return GRAPH_GREATER_THAN_OR_EQUAL_OPERATOR;
            case LESS_THAN:
                return GRAPH_LESS_THAN_OPERATOR;
            case LESS_THAN_OR_EQUAL:
                return GRAPH_LESS_THAN_OR_EQUAL_OPERATOR;
            default:
                throw new IllegalArgumentException("Cannot query by match mode " + matchMode);
        }
    }

    private String resolvePropertyName(final EntityDescriptor entityDescriptor, final Property property, String usernameAttribute) {
        switch (entityDescriptor.getEntityType()) {
            case USER:
                return getUserAttributeName(property, usernameAttribute);
            case GROUP:
                return getGroupAttributeName(property);
            default:
                throw new IllegalArgumentException("Cannot transform entity of type <" + entityDescriptor.getEntityType() + ">");
        }
    }

    private String getGroupAttributeName(final Property property) {
        if (GroupTermKeys.NAME.equals(property)) {
            return GROUPNAME;
        } else if (GroupTermKeys.DESCRIPTION.equals(property)) {
            return DESCRIPTION;
        } else {
            throw new IllegalArgumentException("Cannot query by property " + property);
        }
    }

    private String getUserAttributeName(final Property property, String usernameAttribute) {
        if (UserTermKeys.USERNAME.equals(property)) {
            return usernameAttribute;
        } else if (UserTermKeys.FIRST_NAME.equals(property)) {
            return FIRST_NAME;
        } else if (UserTermKeys.LAST_NAME.equals(property)) {
            return LAST_NAME;
        } else if (UserTermKeys.DISPLAY_NAME.equals(property)) {
            return DISPLAY_NAME;
        } else if (UserTermKeys.EMAIL.equals(property)) {
            return MAIL; //wonky as this can often be empty as it's the address of the user's mailbox
        } else if (UserTermKeys.EXTERNAL_ID.equals(property)) {
            return ID;
        } else if (UserTermKeys.ACTIVE.equals(property)) {
            return ACCOUNT_ENABLED;
        } else {
            throw new IllegalArgumentException("Cannot query by property " + property);
        }
    }

    private <T> T sanitizeOdataValue(final T value) {
        if (value instanceof String) {
            return (T) ((String) value).replace("'", "''");
        } else {
            return value;
        }
    }
}
