package com.atlassian.crowd.directory.cache;

import com.atlassian.crowd.directory.AzureAdDirectory;
import com.atlassian.crowd.directory.AzureMembershipHelper;
import com.atlassian.crowd.directory.ldap.cache.RemoteDirectoryCacheRefresher;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.manager.directory.nestedgroups.NestedGroupsIterator;
import com.atlassian.crowd.manager.directory.nestedgroups.NestedGroupsProvider;
import com.atlassian.crowd.manager.directory.nestedgroups.NestedGroupsProviderBuilder;
import com.atlassian.crowd.model.DirectoryEntities;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.group.ImmutableMembership;
import com.atlassian.crowd.model.group.Membership;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.tuple.Pair;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

public class UserFilterableNonDeltaQueryCacheRefresher extends RemoteDirectoryCacheRefresher {
    private final AzureAdDirectory azureAdDirectory;
    private SyncData syncData;

    public UserFilterableNonDeltaQueryCacheRefresher(AzureAdDirectory azureAdDirectory) {
        super(azureAdDirectory);
        this.azureAdDirectory = azureAdDirectory;
    }

    @Override
    protected List<GroupWithAttributes> findAllRemoteGroups(boolean withAttributes) throws OperationFailedException {
        return ImmutableList.copyOf(getSyncData().getGroups());
    }

    @Override
    protected List<UserWithAttributes> findAllRemoteUsers(boolean withAttributes) throws OperationFailedException {
        return ImmutableList.copyOf(getSyncData().getUsers());
    }

    @Override
    protected Iterable<Membership> getMemberships(Collection groups, boolean isFullSync) throws OperationFailedException {
        return ImmutableList.copyOf(getSyncData().getMemberships());
    }

    private synchronized SyncData getSyncData() throws OperationFailedException {
        if (syncData == null) {
            syncData = computeFullSyncData();
        }
        return syncData;
    }

    private SyncData computeFullSyncData() throws OperationFailedException {
        List<GroupWithAttributes> filteredGroups = azureAdDirectory.getFilteredGroups();
        SyncData syncData = new SyncData();
        syncData.addGroups(filteredGroups);

        AzureMembershipHelper membershipHelper = azureAdDirectory.createMembershipHelper();

        NestedGroupsProvider provider = NestedGroupsProviderBuilder.create()
                .useExternalId()
                .setSingleGroupProvider(
                        (String id) -> {
                            Pair<List<UserWithAttributes>, List<GroupWithAttributes>> children = membershipHelper.getDirectChildren(id);
                            syncData.addMemberships(id, children.getLeft(), children.getRight());
                            return ImmutableList.copyOf(children.getRight());
                        })
                .build();
        List<String> externalIds = filteredGroups.stream().map(GroupWithAttributes::getExternalId).collect(Collectors.toList());
        NestedGroupsIterator.namesIterator(externalIds, true, provider).visitAll();
        return syncData;
    }

    private static final class SyncData {
        final Set<UserWithAttributes> users = new HashSet<>();
        final Set<GroupWithAttributes> groups = new HashSet<>();
        final List<Membership> memberships = new ArrayList<>();
        final Map<String, String> groupIdToName = new HashMap<>();

        public void addGroups(List<GroupWithAttributes> groups) {
            groups.forEach(group -> groupIdToName.put(group.getExternalId(), group.getName()));
            this.groups.addAll(groups);
        }

        public void addMemberships(String groupId, List<UserWithAttributes> users, List<GroupWithAttributes> groups) {
            addGroups(groups);
            this.users.addAll(users);
            memberships.add(new ImmutableMembership(
                    groupIdToName.get(groupId), DirectoryEntities.namesOf(users), DirectoryEntities.namesOf(groups)));
        }

        public Set<UserWithAttributes> getUsers() {
            return users;
        }

        public Set<GroupWithAttributes> getGroups() {
            return groups;
        }

        public List<Membership> getMemberships() {
            return memberships;
        }
    }
}
