package com.atlassian.crowd.directory.cache;

import com.atlassian.crowd.mapper.SimpleObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.io.IOException;
import java.util.Objects;

/**
 * Represents a pair of delta tokens for an Azure AD synchronisation. Meant to be stored in the database as a marshalled
 * string.
 */
public class DeltaQuerySyncTokenHolder {
    private static final Logger log = LoggerFactory.getLogger(DeltaQuerySyncTokenHolder.class);
    private static final SimpleObjectMapper<DeltaQuerySyncTokenHolder> OBJECT_MAPPER =
            new SimpleObjectMapper(DeltaQuerySyncTokenHolder.class);

    @Nullable
    private final String usersDeltaQuerySyncToken;
    @Nullable
    private final String groupsDeltaQuerySyncToken;

    private DeltaQuerySyncTokenHolder() {
        this(null, null);
    }

    public DeltaQuerySyncTokenHolder(String usersDeltaQuerySyncToken, String groupsDeltaQuerySyncToken) {
        this.usersDeltaQuerySyncToken = usersDeltaQuerySyncToken;
        this.groupsDeltaQuerySyncToken = groupsDeltaQuerySyncToken;
    }

    public String getUsersDeltaQuerySyncToken() {
        return usersDeltaQuerySyncToken;
    }

    public String getGroupsDeltaQuerySyncToken() {
        return groupsDeltaQuerySyncToken;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        DeltaQuerySyncTokenHolder that = (DeltaQuerySyncTokenHolder) o;
        return Objects.equals(usersDeltaQuerySyncToken, that.usersDeltaQuerySyncToken) &&
                Objects.equals(groupsDeltaQuerySyncToken, that.groupsDeltaQuerySyncToken);
    }

    @Override
    public int hashCode() {
        return Objects.hash(usersDeltaQuerySyncToken, groupsDeltaQuerySyncToken);
    }

    public String serialize() {
        try {
            return OBJECT_MAPPER.serialize(this);
        } catch (IOException e) {
            log.warn("Cannot serialize synchronisation token obtained from Azure AD. Users sync token: '{}', groups sync token: '{}'",
                    usersDeltaQuerySyncToken, groupsDeltaQuerySyncToken, e);
            return null;
        }
    }

    public static DeltaQuerySyncTokenHolder deserialize(final @Nullable String syncToken) {
        try {
            return OBJECT_MAPPER.deserialize(syncToken);
        } catch (IOException e) {
            log.warn("Cannot perform incremental synchronisation for directory [{}] due to a malformed synchronisation token", syncToken);
            throw new RuntimeException(e);
        }
    }
}
