package com.atlassian.crowd.directory.cache;

import com.atlassian.crowd.directory.AzureAdDirectory;
import com.atlassian.crowd.directory.rest.mapper.DeltaQueryResult;
import com.atlassian.crowd.directory.synchronisation.CacheSynchronisationResult;
import com.atlassian.crowd.directory.synchronisation.PartialSynchronisationResult;
import com.atlassian.crowd.directory.synchronisation.cache.CacheRefresher;
import com.atlassian.crowd.directory.synchronisation.cache.DirectoryCache;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupWithMembershipChanges;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.atlassian.crowd.directory.rest.util.ThrowingMapMergeOperatorUtil.mapUniqueNamesToIds;
import static com.atlassian.crowd.util.EqualityUtil.different;

/**
 * Performs delta queries on Azure Active Directory to facilitate incremental synchronisation. More details can be found
 * at https://developer.microsoft.com/en-us/graph/docs/concepts/delta_query_overview
 * <p>
 * If no delta tokens are present a full synchronisation using Azure AD's delta endpoints will be performed to obtain
 * delta tokens for future incremental syncs.
 * <p>
 * As Azure AD returns a diff of group memberships, this cache refresher will add/remove the memberships specified by
 * Azure AD instead of diffing current memberships with the ones obtained from the remote directory. This also causes a
 * slight overhead due to the need to look up entity names for those memberships as Azure AD only specifies external ids
 * and Crowd needs names. To facilitate this the names for added/changed entities are added into a cache local to a
 * synchronisation and the cache is then populated whenever a lookup is performed.
 * <p>
 * This CacheRefresher uses worker threads for synchronisation. There are two reasons for this:
 * <ul>
 * <li>Increasing performance by concurrently fetching both users and groups</li>
 * <li>Minimising the chance to have an outdated set of users and groups. As these are handled by two separate
 * endpoints and use two separate tokens for tracking their state it's possible that one set will refer to outdated
 * information. For example a user was added to a group and this information was fetched from the groups endpoint.
 * However the user was deleted after that and the deletion was contained in the response from the users endpoint.
 * Concurrent fetching reduces the window for such changes, however they are still possible.</li>
 * </ul>
 */
public class DeltaQueryCacheRefresher implements CacheRefresher {
    private static final Logger log = LoggerFactory.getLogger(DeltaQueryCacheRefresher.class);
    protected final AzureAdDirectory azureAdDirectory;

    public DeltaQueryCacheRefresher(final AzureAdDirectory remoteDirectory) {
        this.azureAdDirectory = remoteDirectory;
    }

    @Override
    public CacheSynchronisationResult synchroniseAll(final DirectoryCache directoryCache) throws OperationFailedException {
        try (BackgroundQueriesProcessor processor = new BackgroundQueriesProcessor(
                "DeltaQueryCacheRefresher-" + azureAdDirectory.getDirectoryId(),
                this::getUsersFromDeltaQuery,
                azureAdDirectory::performGroupsDeltaQuery)) {
            Date syncStartDate = new Date();
            final DeltaQueryResult<UserWithAttributes> allUsers =
                    synchroniseAllUsers(directoryCache, processor.getUsers(), syncStartDate);
            final PartialSynchronisationResult<GroupWithMembershipChanges> allGroups =
                    synchroniseAllGroups(directoryCache, processor.getGroups(), syncStartDate);
            synchroniseAllMemberships(directoryCache, allUsers, allGroups.getResults());
            return new CacheSynchronisationResult(true, new DeltaQuerySyncTokenHolder(
                    allUsers.getSyncToken().orElse(null), allGroups.getSyncToken().orElse(null)).serialize());
        }
    }

    protected DeltaQueryResult<UserWithAttributes> getUsersFromDeltaQuery() throws OperationFailedException {
        return azureAdDirectory.performUsersDeltaQuery();
    }

    protected DeltaQueryResult<UserWithAttributes> getUserChangesFromDeltaQuery(String userSyncToken) throws OperationFailedException {
        return azureAdDirectory.fetchUserChanges(userSyncToken);
    }

    @Override
    public CacheSynchronisationResult synchroniseChanges(final DirectoryCache directoryCache, final @Nullable String syncToken) throws OperationFailedException {
        final Optional<DeltaQuerySyncTokenHolder> validSyncToken = extractSyncToken(syncToken);
        if (!validSyncToken.isPresent()) {
            return CacheSynchronisationResult.FAILURE;
        }
        final DeltaQuerySyncTokenHolder deltaQueryTokens = validSyncToken.get();
        try (BackgroundQueriesProcessor processor = new BackgroundQueriesProcessor(
                "DeltaQueryCacheRefresher-" + azureAdDirectory.getDirectoryId(),
                () -> getUserChangesFromDeltaQuery(deltaQueryTokens.getUsersDeltaQuerySyncToken()),
                () -> azureAdDirectory.fetchGroupChanges(deltaQueryTokens.getGroupsDeltaQuerySyncToken()))) {
            Date syncStartDate = new Date();
            final DeltaQueryResult<UserWithAttributes> mappedUsers =
                    synchroniseUserChanges(directoryCache, processor.getUsers(), syncStartDate);
            final DeltaQueryResult<GroupWithMembershipChanges> mappedGroups =
                    synchroniseGroupChanges(directoryCache, processor.getGroups(), syncStartDate);
            synchroniseMembershipChanges(directoryCache, mappedUsers, mappedGroups.getChangedEntities());

            return new CacheSynchronisationResult(true, new DeltaQuerySyncTokenHolder(
                    mappedUsers.getSyncToken().orElse(null), mappedGroups.getSyncToken().orElse(null)).serialize());
        }
    }

    protected Optional<DeltaQuerySyncTokenHolder> extractSyncToken(String syncToken) {
        if (Strings.isNullOrEmpty(syncToken)) {
            log.info("Synchronisation token not present, full sync of directory [{}] is necessary before incremental sync is possible.",
                    azureAdDirectory.getDirectoryId());
            return Optional.empty();
        }
        return Optional.of(syncToken).map(DeltaQuerySyncTokenHolder::deserialize).filter(this::isValidToken);
    }

    protected boolean isValidToken(final DeltaQuerySyncTokenHolder deltaQueryTokens) {
        if (Strings.isNullOrEmpty(deltaQueryTokens.getGroupsDeltaQuerySyncToken())) {
            log.info("Groups delta token not present, falling back to full sync of directory [{}].",
                    azureAdDirectory.getDirectoryId());
            return false;
        }
        if (Strings.isNullOrEmpty(deltaQueryTokens.getUsersDeltaQuerySyncToken())) {
            log.info("Users delta token not present, falling back to full sync of directory [{}].",
                    azureAdDirectory.getDirectoryId());
            return false;
        }
        return true;
    }

    private void synchroniseAllMemberships(final DirectoryCache directoryCache, final DeltaQueryResult<UserWithAttributes> mappedUsers,
                                           final Collection<GroupWithMembershipChanges> mappedGroups) throws OperationFailedException {
        IdToNameResolver usersResolver = usersResolver(mappedUsers, directoryCache);
        IdToNameResolver groupResolver = groupResolver(mappedGroups, directoryCache);

        for (GroupWithMembershipChanges group : mappedGroups) {
            log.debug("Synchronising memberships for group {}", group.getName());
            if (azureAdDirectory.supportsNestedGroups()) {
                directoryCache.syncGroupMembersForGroup(group, groupResolver.getNames(group.getGroupChildrenIdsToAdd(), true));
            }
            directoryCache.syncUserMembersForGroup(group, usersResolver.getNames(group.getUserChildrenIdsToAdd(), true));
        }
    }

    protected void synchroniseMembershipChanges(final DirectoryCache directoryCache, final DeltaQueryResult<UserWithAttributes> mappedUsers,
                                                final Collection<GroupWithMembershipChanges> mappedGroups) throws OperationFailedException {
        IdToNameResolver usersResolver = usersResolver(mappedUsers, directoryCache);
        IdToNameResolver groupResolver = groupResolver(mappedGroups, directoryCache);

        for (GroupWithMembershipChanges group : mappedGroups) {
            if (azureAdDirectory.supportsNestedGroups()) {
                directoryCache.addGroupMembersForGroup(group, groupResolver.getNames(group.getGroupChildrenIdsToAdd(),  true));
                directoryCache.deleteGroupMembersForGroup(group, groupResolver.getNames(group.getGroupChildrenIdsToDelete(), false));
            }
            directoryCache.addUserMembersForGroup(group, usersResolver.getNames(group.getUserChildrenIdsToAdd(), true));
            directoryCache.deleteUserMembersForGroup(group, usersResolver.getNames(group.getUserChildrenIdsToDelete(), false));
        }
    }

    protected interface IdToNameResolver {
        Set<String> getNames(Set<String> ids, boolean failOnNotResolved) throws OperationFailedException;
    }

    protected interface IdToNameProvider {
        Map<String, String> getIdToNames(Set<String> ids) throws OperationFailedException;
    }

    protected Set<String> getNames(Map<String, String> idToNameCache,
                                   Set<String> idsToResolve,
                                   IdToNameProvider findById,
                                   boolean failOnNotResolved,
                                   String entityType) throws OperationFailedException {
        final Sets.SetView<String> difference = Sets.difference(idsToResolve, idToNameCache.keySet());
        if (!difference.isEmpty()) {
            log.debug("Azure AD reported memberships on {}s that were not modified: {}, "
                    + "trying to resolve them from directory cache", entityType, difference);
            idToNameCache.putAll(findById.getIdToNames(difference));
            if (failOnNotResolved && !difference.isEmpty()) {
                throw new OperationFailedException("Azure AD reported new memberships on " + entityType
                        + "s that were not returned during " + entityType + " sync. "
                        + StringUtils.capitalize(entityType) + "s in question: " + difference);
            }
        }
        return idsToResolve.stream().map(idToNameCache::get).filter(Objects::nonNull).collect(Collectors.toSet());
    }

    protected <T> Map<String, String> mapIdToUniqueNames(
            Collection<T> entities, Function<T, String> idMapper, Function<T, String> nameMapper, String entityName) {
        mapUniqueNamesToIds(entities, nameMapper, idMapper, entityName);  // just to validate name uniqueness
        return entities.stream().collect(Collectors.toMap(idMapper, nameMapper));
    }

    protected IdToNameResolver usersResolver(DeltaQueryResult<UserWithAttributes> mappedUsers, DirectoryCache directoryCache) {
        Map<String, String> cache = mapIdToUniqueNames(mappedUsers.getChangedEntities(), User::getExternalId, User::getName, "user");
        return (idsToResolve, failOnNotResolved) -> getNames(cache, Sets.difference(idsToResolve, mappedUsers.getNamelessEntities()).immutableCopy(), directoryCache::findUsersByExternalIds, failOnNotResolved, "user");
    }

    protected IdToNameResolver groupResolver(
            Collection<GroupWithMembershipChanges> mappedGroups, DirectoryCache directoryCache) {
        Map<String, String> cache = mapIdToUniqueNames(mappedGroups, Group::getExternalId, Group::getName, "group");
        return (idsToResolve, failOnNotResolved) ->
                getNames(cache, idsToResolve, directoryCache::findGroupsByExternalIds, failOnNotResolved, "group");
    }

    protected DeltaQueryResult<UserWithAttributes> synchroniseUserChanges(
            final DirectoryCache directoryCache,
            final DeltaQueryResult<UserWithAttributes> mappedUsers,
            final Date syncStartDate) throws OperationFailedException {
        handleNamelessEntities(mappedUsers, "users");
        directoryCache.deleteCachedUsersByGuid(mappedUsers.getDeletedEntities());
        directoryCache.addOrUpdateCachedUsers(mappedUsers.getChangedEntities(), syncStartDate);
        return mappedUsers;
    }

    protected DeltaQueryResult<GroupWithMembershipChanges> synchroniseGroupChanges(
            final DirectoryCache directoryCache,
            final DeltaQueryResult<GroupWithMembershipChanges> mappedGroups,
            final Date syncStartDate) throws OperationFailedException {
        handleNamelessEntities(mappedGroups, "groups");
        checkNoRenamedGroups(directoryCache, mappedGroups);
        checkNoReaddedGroups(directoryCache, mappedGroups);
        directoryCache.deleteCachedGroupsByGuids(mappedGroups.getDeletedEntities());
        directoryCache.addOrUpdateCachedGroups(mappedGroups.getChangedEntities(), syncStartDate);

        return mappedGroups;
    }

    protected void checkNoRenamedGroups(final DirectoryCache directoryCache, final DeltaQueryResult<GroupWithMembershipChanges> mappedGroups) throws OperationFailedException {
        final Map<String, String> newEntitiesExternalIdsToNames = mappedGroups.getChangedEntities().stream().collect(Collectors.toMap(Group::getExternalId, Group::getName));
        final Map<String, String> externalIdsToNames = directoryCache.findGroupsByExternalIds(newEntitiesExternalIdsToNames.keySet());
        final Set<Map.Entry<String, String>> differences = externalIdsToNames.entrySet().stream().filter(entry -> {
            final String matchingChangedEntity = newEntitiesExternalIdsToNames.get(entry.getKey());
            return matchingChangedEntity != null && different(matchingChangedEntity, entry.getValue());
        }).collect(Collectors.toSet());
        if (!differences.isEmpty()) {
            log.info("Cannot proceed with incremental synchronisation due to groups with known external ids but unknown" +
                    "names, falling back to full synchronisation. Groups in question: [{}]", differences);
            throw new OperationFailedException("Cannot proceed with incremental synchronisation due to renamed groups, " +
                    "falling back to full synchronisation.");
        }
    }

    protected void checkNoReaddedGroups(final DirectoryCache directoryCache, final DeltaQueryResult<GroupWithMembershipChanges> mappedGroups) throws OperationFailedException {
        final Map<String, String> newEntitiesNamesToExternalIds = mapUniqueNamesToIds(mappedGroups.getChangedEntities(), Group::getName, Group::getExternalId, "group");
        final Map<String, String> namesToExternalIds = directoryCache.findGroupsExternalIdsByNames(newEntitiesNamesToExternalIds.keySet());
        final Set<Map.Entry<String, String>> differences = namesToExternalIds.entrySet().stream().filter(entry -> {
            final String matchingChangedEntity = newEntitiesNamesToExternalIds.get(entry.getKey());
            return matchingChangedEntity != null && different(matchingChangedEntity, entry.getValue()) && !mappedGroups.getDeletedEntities().contains(entry.getValue());
        }).collect(Collectors.toSet());
        if (!differences.isEmpty()) {
            log.info("Cannot proceed with incremental synchronisation due to groups readded with known names/duplicates" +
                    ", falling back to full synchronisation. Groups in question: [{}]", differences);
            throw new OperationFailedException("Cannot proceed with incremental synchronisation due to readded/duplicate groups, " +
                    "falling back to full synchronisation.");
        }
    }

    private DeltaQueryResult<UserWithAttributes> synchroniseAllUsers(
            final DirectoryCache directoryCache, final DeltaQueryResult<UserWithAttributes> mappedUsers, Date syncStartDate) throws OperationFailedException {
        handleNamelessEntities(mappedUsers, "users");
        directoryCache.deleteCachedUsersNotIn(mappedUsers.getChangedEntities(), syncStartDate);
        directoryCache.addOrUpdateCachedUsers(mappedUsers.getChangedEntities(), syncStartDate);
        return mappedUsers;
    }

    private PartialSynchronisationResult<GroupWithMembershipChanges> synchroniseAllGroups(
            final DirectoryCache directoryCache,
            DeltaQueryResult<GroupWithMembershipChanges> mappedGroups,
            Date syncStartDate) throws OperationFailedException {
        handleNamelessEntities(mappedGroups, "groups");
        directoryCache.deleteCachedGroupsNotInByExternalId(mappedGroups.getChangedEntities(), syncStartDate);
        directoryCache.addOrUpdateCachedGroups(mappedGroups.getChangedEntities(), syncStartDate);
        return new PartialSynchronisationResult<>(mappedGroups.getChangedEntities(), mappedGroups.getSyncToken().orElse(null));
    }

    protected <T> void handleNamelessEntities(DeltaQueryResult<T> mappedEntities, String entityType) {
        final Sets.SetView<String> difference = Sets.difference(mappedEntities.getNamelessEntities(), mappedEntities.getDeletedEntities());
        if (!difference.isEmpty()) {
            log.warn("Azure AD returned the following {} without ids: {}", entityType, difference);
        }
    }
}
