package com.atlassian.crowd.directory.cache;

import com.atlassian.crowd.directory.AzureAdDirectory;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.DirectoryType;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.model.config.AzureGroupFiltersConfiguration;
import com.atlassian.crowd.model.directory.ImmutableDirectory;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.type.JavaType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Collections;
import java.util.Set;

public class AzureGroupFilterProcessor {

    private static final Logger log = LoggerFactory.getLogger(AzureGroupFilterProcessor.class);
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private final DirectoryManager directoryManager;

    public AzureGroupFilterProcessor(DirectoryManager directoryManager) {
        this.directoryManager = directoryManager;
    }

    public static Set<String> getGroupNames(String attributeValue) {
        if (StringUtils.isBlank(attributeValue)) {
            return Collections.emptySet();
        }

        final JavaType setOfStrings = OBJECT_MAPPER.getTypeFactory().constructCollectionType(Set.class, String.class);
        try {
            return ImmutableSet.copyOf((Set<String>) OBJECT_MAPPER.readValue(attributeValue, setOfStrings));
        } catch (Exception e) {
            log.error("Unable to parse attribute JSON value containing list of groups to be filtered for Azure AD: {}",
                    attributeValue);
            throw new RuntimeException("Invalid value for list of groups to be filtered");
        }
    }

    public static String prepareAttribute(Set<String> groupNames) {
        try {
            return OBJECT_MAPPER.writeValueAsString(groupNames);
        } catch (IOException e) {
            log.error("Unable to serialize list of group external Ids to filter", e);
            throw new RuntimeException("Invalid value for list of groups to be filtered");
        }
    }

    public AzureGroupFiltersConfiguration getConfiguration(long directoryId) throws DirectoryNotFoundException {
        final Directory directory = loadAzureDirectoryOrElseThrow(directoryId);
        final Set<String> groupsToFilter = getGroupNames(directory.getValue(AzureAdDirectory.FILTERED_GROUPS_ATTRIBUTE));
        final boolean enabled = Boolean.parseBoolean(directory.getAttributes()
                .getOrDefault(AzureAdDirectory.GROUP_FILTERING_ENABLED_ATTRIBUTE, Boolean.FALSE.toString()));
        return new AzureGroupFiltersConfiguration(enabled, groupsToFilter);
    }

    public void configureGroupFilter(long directoryId, AzureGroupFiltersConfiguration configuration) throws DirectoryNotFoundException {
        final Directory directory = loadAzureDirectoryOrElseThrow(directoryId);
        directoryManager.updateDirectory(ImmutableDirectory.builder(directory)
                .setAttribute(AzureAdDirectory.GROUP_FILTERING_ENABLED_ATTRIBUTE, Boolean.toString(configuration.isEnabled()))
                .setAttribute(AzureAdDirectory.FILTERED_GROUPS_ATTRIBUTE, prepareAttribute(configuration.getGroupsNames()))
                .build());
    }

    private Directory loadAzureDirectoryOrElseThrow(long directoryId) throws DirectoryNotFoundException {
        final Directory directory = directoryManager.findDirectoryById(directoryId);
        if (directory.getType() != DirectoryType.AZURE_AD) {
            throw new IllegalArgumentException("Cannot configure filterable groups for non-azure active directory");
        }

        return directory;
    }

}
