package com.atlassian.crowd.directory;

import com.atlassian.crowd.directory.query.FetchMode;
import com.atlassian.crowd.directory.query.GraphQuery;
import com.atlassian.crowd.directory.query.MicrosoftGraphQueryTranslator;
import com.atlassian.crowd.directory.query.ODataFilter;
import com.atlassian.crowd.directory.query.ODataSelect;
import com.atlassian.crowd.directory.query.ODataTop;
import com.atlassian.crowd.directory.rest.AzureAdPagingWrapper;
import com.atlassian.crowd.directory.rest.AzureAdRestClient;
import com.atlassian.crowd.directory.rest.entity.group.GraphGroup;
import com.atlassian.crowd.directory.rest.entity.membership.DirectoryObject;
import com.atlassian.crowd.directory.rest.mapper.AzureAdRestEntityMapper;
import com.atlassian.crowd.directory.rest.util.MembershipFilterUtil;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.function.ExceptionTranslators;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.group.ImmutableMembership;
import com.atlassian.crowd.model.group.Membership;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.search.EntityDescriptor;
import com.google.common.collect.Iterators;
import org.apache.commons.lang3.tuple.Pair;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.atlassian.crowd.directory.rest.util.ThrowingMapMergeOperatorUtil.mapUniqueNamesToIds;

/**
 * Helper class to fetch membership data.
 */
public class AzureMembershipHelper {
    private AzureAdRestClient restClient;
    private AzureAdPagingWrapper pagingWrapper;
    private MicrosoftGraphQueryTranslator queryTranslator;
    private AzureAdRestEntityMapper restEntityMapper;
    private AzureAdDirectory directory;

    public AzureMembershipHelper(AzureAdRestClient restClient, AzureAdPagingWrapper pagingWrapper, MicrosoftGraphQueryTranslator queryTranslator, AzureAdRestEntityMapper restEntityMapper, AzureAdDirectory directory) {
        this.restClient = restClient;
        this.pagingWrapper = pagingWrapper;
        this.queryTranslator = queryTranslator;
        this.restEntityMapper = restEntityMapper;
        this.directory = directory;
    }

    /**
     * @return iterator of memberships; sub-groups will be included only if nested groups are enabled
     * @throws OperationFailedException
     */
    public Iterator<Membership> membershipIterator() throws OperationFailedException {
        final ODataSelect select = queryTranslator.resolveAzureAdColumnsForSingleEntityTypeQuery(EntityDescriptor.group(), FetchMode.NAME_AND_ID);
        final GraphQuery query = new GraphQuery(ODataFilter.EMPTY, select, 0, ODataTop.FULL_PAGE);
        final List<GraphGroup> groups = pagingWrapper.fetchAllResults(restClient.searchGroups(query));
        final Map<String, String> groupNamesToIds = mapUniqueNamesToIds(groups, GraphGroup::getDisplayName, GraphGroup::getId, "group");

        final Function<Map.Entry<String, String>, Membership> lookUpMembers = ExceptionTranslators.toRuntimeException(
                (entry) -> getMembership(entry.getKey(), entry.getValue()),
                Membership.MembershipIterationException::new);

        return Iterators.transform(groupNamesToIds.entrySet().iterator(), lookUpMembers::apply);
    }

    /**
     * Returns direct children of a group.
     * @param groupId id of a group
     * @return direct children of a group; sub-groups will be returned only if nested groups are enabled
     * @throws OperationFailedException
     */
    public Pair<List<UserWithAttributes>, List<GroupWithAttributes>> getDirectChildren(String groupId) throws OperationFailedException {
        ODataSelect select = queryTranslator.translateColumnsForUsersAndGroupsQuery(FetchMode.FULL);
        Pair<List<DirectoryObject>, List<DirectoryObject>> children = getChildrenUsersAndGroups(groupId, select);
        return Pair.of(mapTo(children.getLeft(), UserWithAttributes.class), mapTo(children.getRight(), GroupWithAttributes.class));
    }

    private Pair<List<DirectoryObject>, List<DirectoryObject>> getChildrenUsersAndGroups(
            String groupId, ODataSelect select) throws OperationFailedException {
        final List<DirectoryObject> children = pagingWrapper.fetchAllResults(restClient.getDirectChildrenOfGroup(groupId, select));
        return Pair.of(children.stream().filter(MembershipFilterUtil::isUser).collect(Collectors.toList()),
                !directory.supportsNestedGroups() ? Collections.emptyList()
                        : children.stream().filter(MembershipFilterUtil::isGroup).collect(Collectors.toList()));
    }

    private Membership getMembership(String groupName, String groupId) throws OperationFailedException {
        Pair<List<DirectoryObject>, List<DirectoryObject>> children = getChildrenUsersAndGroups(
                groupId, queryTranslator.translateColumnsForUsersAndGroupsQuery(FetchMode.NAME));
        return new ImmutableMembership(
                groupName, mapTo(children.getLeft(), String.class), mapTo(children.getRight(), String.class));
    }

    private <T> List<T> mapTo(List<DirectoryObject> list, Class<? extends T> cls) {
        return list.stream().map(mapTo(cls)).collect(Collectors.toList());
    }

    private <T> Function<DirectoryObject, T> mapTo(Class<? extends T> cls) {
        final String alternateUsernameAttribute = directory.getAlternativeUsernameAttribute();
        return o -> restEntityMapper.mapDirectoryObject(o, cls, directory.getDirectoryId(), alternateUsernameAttribute);
    }
}
