package com.atlassian.crowd.directory;

import com.atlassian.crowd.directory.authentication.UserCredentialVerifier;
import com.atlassian.crowd.directory.authentication.UserCredentialVerifierFactory;
import com.atlassian.crowd.directory.cache.AzureGroupFilterProcessor;
import com.atlassian.crowd.directory.ldap.LDAPPropertiesMapper;
import com.atlassian.crowd.directory.query.FetchMode;
import com.atlassian.crowd.directory.query.GraphQuery;
import com.atlassian.crowd.directory.query.MicrosoftGraphDeltaToken;
import com.atlassian.crowd.directory.query.MicrosoftGraphQueryTranslator;
import com.atlassian.crowd.directory.query.ODataExpand;
import com.atlassian.crowd.directory.query.ODataFilter;
import com.atlassian.crowd.directory.query.ODataSelect;
import com.atlassian.crowd.directory.query.ODataTop;
import com.atlassian.crowd.directory.rest.AzureAdPagingWrapper;
import com.atlassian.crowd.directory.rest.AzureAdRestClient;
import com.atlassian.crowd.directory.rest.AzureAdRestClientFactory;
import com.atlassian.crowd.directory.rest.delta.GraphDeltaQueryResult;
import com.atlassian.crowd.directory.rest.endpoint.AzureApiUriResolver;
import com.atlassian.crowd.directory.rest.endpoint.AzureApiUriResolverFactory;
import com.atlassian.crowd.directory.rest.entity.GraphDirectoryObjectList;
import com.atlassian.crowd.directory.rest.entity.PageableGraphList;
import com.atlassian.crowd.directory.rest.entity.delta.GraphDeltaQueryGroup;
import com.atlassian.crowd.directory.rest.entity.delta.GraphDeltaQueryUser;
import com.atlassian.crowd.directory.rest.entity.group.GraphGroup;
import com.atlassian.crowd.directory.rest.entity.group.GraphGroupList;
import com.atlassian.crowd.directory.rest.entity.membership.DirectoryObject;
import com.atlassian.crowd.directory.rest.entity.user.GraphUser;
import com.atlassian.crowd.directory.rest.mapper.AzureAdRestEntityMapper;
import com.atlassian.crowd.directory.rest.mapper.DeltaQueryResult;
import com.atlassian.crowd.directory.rest.util.MembershipFilterUtil;
import com.atlassian.crowd.directory.synchronisation.Defaults;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.embedded.api.SearchRestriction;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.embedded.spi.DcLicenseChecker;
import com.atlassian.crowd.exception.ExpiredCredentialException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.InactiveAccountException;
import com.atlassian.crowd.exception.InvalidAuthenticationException;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.exception.InvalidGroupException;
import com.atlassian.crowd.exception.InvalidMembershipException;
import com.atlassian.crowd.exception.InvalidUserException;
import com.atlassian.crowd.exception.MembershipAlreadyExistsException;
import com.atlassian.crowd.exception.MembershipNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.OperationNotSupportedException;
import com.atlassian.crowd.exception.ReadOnlyGroupException;
import com.atlassian.crowd.exception.UserAlreadyExistsException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.avatar.AvatarReference;
import com.atlassian.crowd.model.directory.DirectoryImpl;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplate;
import com.atlassian.crowd.model.group.GroupTemplateWithAttributes;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.group.GroupWithMembershipChanges;
import com.atlassian.crowd.model.group.Membership;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.crowd.model.user.UserTemplateWithAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.builder.Restriction;
import com.atlassian.crowd.search.query.QueryUtils;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.entity.restriction.BooleanRestriction;
import com.atlassian.crowd.search.query.entity.restriction.NullRestriction;
import com.atlassian.crowd.search.query.entity.restriction.NullRestrictionImpl;
import com.atlassian.crowd.search.query.entity.restriction.PropertyRestriction;
import com.atlassian.crowd.search.query.entity.restriction.constants.GroupTermKeys;
import com.atlassian.crowd.search.query.entity.restriction.constants.UserTermKeys;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.crowd.search.util.QuerySplitter;
import com.atlassian.crowd.search.util.ResultsAggregator;
import com.atlassian.crowd.search.util.ResultsAggregators;
import com.atlassian.crowd.util.AttributeUtil;
import com.atlassian.crowd.util.BoundedCount;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.time.Duration;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import static com.atlassian.crowd.directory.SynchronisableDirectoryProperties.CONNECTION_TIMEOUT_IN_MILLISECONDS;
import static com.atlassian.crowd.directory.SynchronisableDirectoryProperties.READ_TIMEOUT_IN_MILLISECONDS;
import static com.atlassian.crowd.directory.rest.util.MembershipFilterUtil.isGroup;

/**
 * Microsoft Azure Active Directory connector
 */
public class AzureAdDirectory implements RemoteDirectory {

    private static Logger logger = LoggerFactory.getLogger(AzureAdDirectory.class);

    public static final String WEBAPP_CLIENT_ID_ATTRIBUTE = "AZURE_AD_WEBAPP_CLIENT_ID";
    public static final String WEBAPP_CLIENT_SECRET_ATTRIBUTE = "AZURE_AD_WEBAPP_CLIENT_SECRET";
    public static final String TENANT_ID_ATTRIBUTE = "AZURE_AD_TENANT_ID";
    public static final String NATIVE_APP_ID_ATTRIBUTE = "AZURE_AD_NATIVE_AP_IDD";
    public static final String GRAPH_API_ENDPOINT_ATTRIBUTE = "AZURE_AD_GRAPH_API_ENDPOINT";
    public static final String AUTHORITY_API_ENDPOINT_ATTRIBUTE = "AZURE_AD_AUTHORITY_API_ENDPOINT";
    public static final String REGION_ATTRIBUTE = "AZURE_AD_REGION";
    public static final String CUSTOM_REGION_ATTRIBUTE_VALUE = "CUSTOM";
    public static final String FILTERED_GROUPS_ATTRIBUTE = "AZURE_AD_FILTERED_GROUPS";
    public static final String GROUP_FILTERING_ENABLED_ATTRIBUTE = "GROUP_FILTERING_ENABLED";
    public static final String NOT_IMPLEMENTED = "Azure Active Directory support is Read-only";
    public static final String ALTERNATIVE_USERNAME_ATTRIBUTE = "ALTERNATIVE_USERNAME_ATTRIBUTE";

    public static final int MAX_RESTRICTIONS_PER_QUERY = 10;

    private final AzureAdRestClientFactory restClientFactory;
    private final MicrosoftGraphQueryTranslator graphQueryTranslator;
    private final AzureAdRestEntityMapper restEntityMapper;
    private final UserCredentialVerifierFactory credentialVerifierFactory;
    private final AzureApiUriResolverFactory endpointDataProviderFactory;
    private final DcLicenseChecker dcLicenseChecker;
    private AzureApiUriResolver endpointDataProvider;
    private Supplier<UserCredentialVerifier> userCredentialVerifier;
    private Supplier<AzureAdRestClient> azureAdRestClient;
    private Supplier<AzureAdPagingWrapper> azureAdPagingWrapper;
    private AttributeValuesHolder attributes;
    private long directoryId;
    private boolean supportsNestedGroups;
    private boolean localGroupsEnabled;

    public AzureAdDirectory(
            final AzureAdRestClientFactory restClientFactory,
            final MicrosoftGraphQueryTranslator graphQueryTranslator,
            final AzureAdRestEntityMapper restEntityMapper,
            final UserCredentialVerifierFactory credentialVerifierFactory,
            final AzureApiUriResolverFactory endpointDataProviderFactory,
            final DcLicenseChecker dcLicenseChecker) {
        this.restClientFactory = restClientFactory;
        this.graphQueryTranslator = graphQueryTranslator;
        this.restEntityMapper = restEntityMapper;
        this.credentialVerifierFactory = credentialVerifierFactory;
        this.endpointDataProviderFactory = endpointDataProviderFactory;
        this.dcLicenseChecker = dcLicenseChecker;
    }

    @Nullable
    @Override
    public Set<String> getValues(final String key) {
        return attributes.getValues(key);
    }

    @Nullable
    @Override
    public String getValue(final String key) {
        return attributes.getValue(key);
    }

    @Override
    public Set<String> getKeys() {
        return attributes.getKeys();
    }

    @Override
    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    @Override
    public long getDirectoryId() {
        return directoryId;
    }

    @Override
    public void setDirectoryId(final long directoryId) {
        this.directoryId = directoryId;
    }

    @Nonnull
    @Override
    public String getDescriptiveName() {
        return "Microsoft Azure Active Directory";
    }

    @Override
    public void setAttributes(final Map<String, String> attributes) {
        this.attributes = new AttributeValuesHolder(attributes);
        final String webappClientId = attributes.get(WEBAPP_CLIENT_ID_ATTRIBUTE);
        final String webappClientSecret = attributes.get(WEBAPP_CLIENT_SECRET_ATTRIBUTE);
        final String tenantId = attributes.get(TENANT_ID_ATTRIBUTE);
        final String nativeClientId = attributes.get(NATIVE_APP_ID_ATTRIBUTE);
        final Duration connectionTimeout = AttributeUtil.safeParseDurationMillis(attributes.get(CONNECTION_TIMEOUT_IN_MILLISECONDS), Defaults.CONNECTION_TIMEOUT);
        final Duration readTimeout = AttributeUtil.safeParseDurationMillis(attributes.get(READ_TIMEOUT_IN_MILLISECONDS), Defaults.READ_TIMEOUT);
        endpointDataProvider = endpointDataProviderFactory.getEndpointDataProviderForDirectory(this);
        supportsNestedGroups = Boolean.parseBoolean(attributes.get(DirectoryImpl.ATTRIBUTE_KEY_USE_NESTED_GROUPS));
        localGroupsEnabled = Boolean.parseBoolean(attributes.get(LDAPPropertiesMapper.LOCAL_GROUPS));
        azureAdRestClient = Suppliers.memoize(() -> restClientFactory.create(
                webappClientId,
                webappClientSecret,
                tenantId,
                this.endpointDataProvider,
                connectionTimeout.toMillis(),
                readTimeout.toMillis())
        )::get;
        azureAdPagingWrapper = Suppliers.memoize(() -> restClientFactory.create(getRestClient()))::get;
        userCredentialVerifier = Suppliers.memoize(() -> credentialVerifierFactory.create(endpointDataProvider, nativeClientId, tenantId))::get;
    }

    @Nonnull
    @Override
    public User findUserByName(final String name) throws UserNotFoundException, OperationFailedException {
        final EntityQuery<User> query = QueryBuilder.queryFor(User.class, EntityDescriptor.user())
                .with(Restriction.on(UserTermKeys.USERNAME).exactlyMatching(name))
                .returningAtMost(2);
        return getSingleResult(searchUsersWithFallback(query, true), () -> new UserNotFoundException(name));
    }

    @Nonnull
    @Override
    public UserWithAttributes findUserWithAttributesByName(final String name) throws UserNotFoundException, OperationFailedException {
        return UserTemplateWithAttributes.toUserWithNoAttributes(findUserByName(name));
    }

    @Nonnull
    @Override
    public User findUserByExternalId(final String externalId) throws UserNotFoundException, OperationFailedException {
        final EntityQuery<User> query = QueryBuilder.queryFor(User.class, EntityDescriptor.user())
                .with(Restriction.on(UserTermKeys.EXTERNAL_ID).exactlyMatching(externalId))
                .returningAtMost(2);
        return getSingleResult(searchUsersInternal(query), () -> UserNotFoundException.forExternalId(externalId));
    }

    @Nonnull
    @Override
    public User authenticate(final String name, final PasswordCredential credential) throws UserNotFoundException, InactiveAccountException, InvalidAuthenticationException, ExpiredCredentialException, OperationFailedException {
        final User user = findUserByName(name);
        getUserCredentialVerifier().checkUserCredential(name, credential);
        return user;
    }

    @Nonnull
    @Override
    public User addUser(final UserTemplate user, final PasswordCredential credential) throws InvalidUserException, InvalidCredentialException, UserAlreadyExistsException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Override
    public UserWithAttributes addUser(final UserTemplateWithAttributes user, final PasswordCredential credential) throws InvalidUserException, InvalidCredentialException, UserAlreadyExistsException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Nonnull
    @Override
    public User updateUser(final UserTemplate user) throws InvalidUserException, UserNotFoundException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Override
    public void updateUserCredential(final String username, final PasswordCredential credential) throws UserNotFoundException, InvalidCredentialException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Nonnull
    @Override
    public User renameUser(final String oldName, final String newName) throws UserNotFoundException, InvalidUserException, UserAlreadyExistsException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Override
    public void storeUserAttributes(final String username, final Map<String, Set<String>> attributes) throws UserNotFoundException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Override
    public void removeUserAttributes(final String username, final String attributeName) throws UserNotFoundException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Override
    public void removeUser(final String name) throws UserNotFoundException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Nonnull
    @Override
    public <T> List<T> searchUsers(final EntityQuery<T> query) throws OperationFailedException {
        QueryUtils.checkAssignableFrom(query.getReturnType(), String.class, User.class);

        return QuerySplitter.batchConditionsIfNeeded(query, q -> searchUsersWithFallback(q, false), MAX_RESTRICTIONS_PER_QUERY);
    }

    private <T> List<T> searchUsersWithFallback(final EntityQuery<T> query, boolean fallbackOnlyIfEmpty) throws OperationFailedException {
        String alternativeUsernameAttribute = getAlternativeUsernameAttribute();
        if (StringUtils.isEmpty(alternativeUsernameAttribute) || !hasUsernameRestriction(query.getSearchRestriction())) {
            return searchUsersInternal(query);
        }
        if (fallbackOnlyIfEmpty) {
            Preconditions.checkArgument(query.getStartIndex() == 0);
            List<T> originalResults = searchWithUpnFiltering(query, alternativeUsernameAttribute, true);
            return originalResults.isEmpty() ? searchWithUpnFiltering(query, alternativeUsernameAttribute, false) : originalResults;
        }
        ResultsAggregator<T> aggregator = ResultsAggregators.with(query);
        aggregator.addAll(searchWithUpnFiltering(query.withAllResults(), alternativeUsernameAttribute, true));
        aggregator.addAll(searchWithUpnFiltering(query.withAllResults(), alternativeUsernameAttribute, false));
        return aggregator.constrainResults();
    }

    private <T> List<T> searchWithUpnFiltering(EntityQuery<T> fallbackQuery, String alternativeUsernameAttribute, boolean matchUpn) throws OperationFailedException {
        final String matchAttribute = matchUpn ? MicrosoftGraphQueryTranslator.USERNAME : alternativeUsernameAttribute;
        return searchGraphUsers(fallbackQuery.withReturnType(User.class), matchAttribute).stream()
                .filter(u -> matches(u, matchUpn, alternativeUsernameAttribute))
                .map(u -> restEntityMapper.mapUser(u, fallbackQuery.getReturnType(), directoryId, alternativeUsernameAttribute))
                .collect(Collectors.toList());
    }

    private boolean matches(GraphUser user, boolean matchUpn, String alternativeUsernameAttribute) {
        final String mappedUsername = restEntityMapper.getUsername(user, alternativeUsernameAttribute);
        boolean upnAsUsername = Objects.equals(user.getUserPrincipalName(), mappedUsername);
        return matchUpn == upnAsUsername;
    }

    private boolean hasUsernameRestriction(SearchRestriction restriction) {
        if (restriction instanceof PropertyRestriction) {
            return ((PropertyRestriction<?>) restriction).getProperty().equals(UserTermKeys.USERNAME);
        } else if (restriction instanceof BooleanRestriction) {
            return ((BooleanRestriction) restriction).getRestrictions().stream().anyMatch(this::hasUsernameRestriction);
        }
        return false;
    }

    private <T> List<T> searchUsersInternal(final EntityQuery<T> query) throws OperationFailedException {
        List<GraphUser> graphUsers = searchGraphUsers(convert(query), query);
        return restEntityMapper.mapUsers(graphUsers, query.getReturnType(), getDirectoryId(), getAlternativeUsernameAttribute());
    }

    private List<GraphUser> searchGraphUsers(final EntityQuery<?> query, String usernameAttribute) throws OperationFailedException {
        return searchGraphUsers(graphQueryTranslator.convert(query, usernameAttribute), query);
    }

    private List<GraphUser> searchGraphUsers(final GraphQuery graphQuery, final EntityQuery<?> query) throws OperationFailedException {
        return getAzureAdPagingWrapper().fetchAppropriateAmountOfResults(
                getRestClient().searchUsers(graphQuery), query.getStartIndex(), query.getMaxResults());
    }

    @Nonnull
    @Override
    public Group findGroupByName(final String name) throws GroupNotFoundException, OperationFailedException {
        final EntityQuery<Group> query = QueryBuilder.queryFor(Group.class, EntityDescriptor.group())
                .with(Restriction.on(GroupTermKeys.NAME).exactlyMatching(name))
                .returningAtMost(2);
        final GraphGroupList graphGroupList = getRestClient().searchGroups(convert(query));
        validateSingleResult(graphGroupList, () -> new GroupNotFoundException(name));
        return Iterables.getOnlyElement(restEntityMapper.mapGroups(graphGroupList, query.getReturnType(), directoryId));
    }

    @Nonnull
    @Override
    public GroupWithAttributes findGroupWithAttributesByName(final String name) throws GroupNotFoundException, OperationFailedException {
        return GroupTemplateWithAttributes.ofGroupWithNoAttributes(findGroupByName(name));
    }

    @Nonnull
    @Override
    public Group addGroup(final GroupTemplate group) throws InvalidGroupException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Nonnull
    @Override
    public Group updateGroup(final GroupTemplate group) throws InvalidGroupException, GroupNotFoundException, ReadOnlyGroupException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Nonnull
    @Override
    public Group renameGroup(final String oldName, final String newName) throws GroupNotFoundException, InvalidGroupException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Override
    public void storeGroupAttributes(final String groupName, final Map<String, Set<String>> attributes) throws GroupNotFoundException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Override
    public void removeGroupAttributes(final String groupName, final String attributeName) throws GroupNotFoundException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Override
    public void removeGroup(final String name) throws GroupNotFoundException, ReadOnlyGroupException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Nonnull
    @Override
    public <T> List<T> searchGroups(final EntityQuery<T> query) throws OperationFailedException {
        QueryUtils.checkAssignableFrom(query.getReturnType(), String.class, Group.class);
        return QuerySplitter.batchConditionsIfNeeded(query, this::searchGroupsSplit, MAX_RESTRICTIONS_PER_QUERY);
    }

    private <T> List<T> searchGroupsSplit(final EntityQuery<T> query) throws OperationFailedException {
        final GraphGroupList graphGroupsList = getRestClient().searchGroups(convert(query));
        final List<GraphGroup> results = getAzureAdPagingWrapper().fetchAppropriateAmountOfResults(graphGroupsList, query.getStartIndex(), query.getMaxResults());
        return restEntityMapper.mapGroups(results, query.getReturnType(), getDirectoryId());
    }

    @Override
    public boolean isUserDirectGroupMember(final String username, final String groupName) throws OperationFailedException {
        final ODataSelect select = graphQueryTranslator.resolveAzureAdColumnsForSingleEntityTypeQuery(EntityDescriptor.group(), FetchMode.NAME);
        final String externalIdOrName = StringUtils.isEmpty(getAlternativeUsernameAttribute()) ? username : fetchExternalIdOfUser(username)
                .orElseThrow(() -> new OperationFailedException(new UserNotFoundException(username)));
        final Optional<DirectoryObject> maybeGroup = getAzureAdPagingWrapper().pageForElement(
                getRestClient().getDirectParentsOfUser(externalIdOrName, select),
                withGroup(groupName));
        return maybeGroup.isPresent();
    }

    private Predicate<DirectoryObject> withGroup(final String groupName) {
        return g -> isGroup(g) && g.getDisplayName().equals(groupName);
    }

    @Override
    public boolean isGroupDirectGroupMember(final String childGroup, final String parentGroup) throws OperationFailedException {
        final String groupExternalId = fetchExternalIdOfGroup(childGroup)
                .orElseThrow(() -> new OperationFailedException(new GroupNotFoundException(childGroup)));
        final ODataSelect select = graphQueryTranslator.resolveAzureAdColumnsForSingleEntityTypeQuery(EntityDescriptor.group(), FetchMode.NAME);
        final Optional<DirectoryObject> maybeGroup = getAzureAdPagingWrapper().pageForElement(getRestClient().getDirectParentsOfGroup(groupExternalId, select),
                withGroup(parentGroup));
        return maybeGroup.isPresent();
    }

    @Nonnull
    @Override
    public BoundedCount countDirectMembersOfGroup(final String groupName, final int querySizeHint) throws OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Override
    public void addUserToGroup(final String username, final String groupName) throws GroupNotFoundException, UserNotFoundException, ReadOnlyGroupException, OperationFailedException, MembershipAlreadyExistsException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Override
    public void addGroupToGroup(final String childGroup, final String parentGroup) throws GroupNotFoundException, InvalidMembershipException, ReadOnlyGroupException, OperationFailedException, MembershipAlreadyExistsException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Override
    public void removeUserFromGroup(final String username, final String groupName) throws GroupNotFoundException, UserNotFoundException, MembershipNotFoundException, ReadOnlyGroupException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Override
    public void removeGroupFromGroup(final String childGroup, final String parentGroup) throws GroupNotFoundException, InvalidMembershipException, MembershipNotFoundException, ReadOnlyGroupException, OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Nonnull
    @Override
    public <T> List<T> searchGroupRelationships(final MembershipQuery<T> query) throws OperationFailedException {
        Preconditions.checkArgument(query.getSearchRestriction() == NullRestrictionImpl.INSTANCE,
                "Azure AD membership queries do not support search restrictions.");
        if (query.getEntityToReturn() == EntityDescriptor.group()
                && query.getEntityToMatch() == EntityDescriptor.group()
                && !supportsNestedGroups) {
            return Collections.emptyList();
        }
        if (query.getReturnType() == String.class && query.getEntityToReturn().equals(EntityDescriptor.user())) {
            String alternativeUsernameAttribute = getAlternativeUsernameAttribute();
            if (StringUtils.isNotBlank(alternativeUsernameAttribute)) {
                return (List<T>) searchGroupRelationships(query.withReturnType(User.class)).stream()
                        .map(User::getName)
                        .collect(Collectors.toList());
            }
        }

        final ODataSelect select = graphQueryTranslator.resolveAzureAdColumnsForSingleEntityTypeQuery(query.getEntityToReturn(), query.getReturnType());

        if (query.isFindChildren()) {
            Preconditions.checkArgument(query.getEntityToMatch() == EntityDescriptor.group(),
                    "Cannot search for children of entities other than groups");

            return fetchAllResults(query, this::fetchExternalIdOfGroup, id -> getRestClient().getDirectChildrenOfGroup(id, select));
        } else {
            Preconditions.checkArgument(query.getEntityToReturn() == EntityDescriptor.group(),
                    "Cannot search for parents of other types than groups");

            if (query.getEntityToMatch() == EntityDescriptor.user()) {
                // Fetching memberships works for UPN and external id. If username comes from alternative username
                // attribute we need to fetch external id first.
                // If there is no alternative username attribute it means username = UPN, so we can safely use it.
                ExternalIdResolver resolver = StringUtils.isEmpty(getAlternativeUsernameAttribute())
                        ? Optional::of
                        : this::fetchExternalIdOfUser;
                return fetchAllResults(query, resolver, id -> getRestClient().getDirectParentsOfUser(id, select));
            } else if (query.getEntityToMatch() == EntityDescriptor.group()) {
                return fetchAllResults(query, this::fetchExternalIdOfGroup, id -> getRestClient().getDirectParentsOfGroup(id, select));
            } else {
                throw new IllegalArgumentException("Unsupported entity type " + query.getEntityToMatch());
            }
        }
    }

    private <T> List<T> fetchAllResults(final MembershipQuery<T> query,
                                        ExternalIdResolver idResolver,
                                        PageProvider provider)
            throws OperationFailedException {
        final String alternativeUsernameAttribute = getAlternativeUsernameAttribute();
        Predicate<DirectoryObject> filter = getDirectoryObjectFilter(query);
        ResultsAggregator<T> aggregator = ResultsAggregators.with(query);
        for (final String name : query.getEntityNamesToMatch()) {
            final Optional<String> externalId = idResolver.getExternalId(name);
            if (externalId.isPresent()) {
                List<DirectoryObject> results = getAzureAdPagingWrapper().fetchAllMatchingResults(
                        provider.getPage(externalId.get()), filter);
                aggregator.addAll(restEntityMapper.mapDirectoryObjects(results, query.getReturnType(), directoryId, alternativeUsernameAttribute));
            }
        }
        return aggregator.constrainResults();
    }

    interface ExternalIdResolver {
        Optional<String> getExternalId(String name) throws OperationFailedException;
    }

    interface PageProvider {
        GraphDirectoryObjectList getPage(String externalId) throws OperationFailedException;
    }

    @Override
    public void testConnection() throws OperationFailedException {
        final GraphQuery graphQuery = convert(QueryBuilder.queryFor(String.class, EntityDescriptor.user()).returningAtMost(1));
        azureAdRestClient.get().searchUsers(graphQuery);
    }

    @Override
    public boolean supportsInactiveAccounts() {
        return true;
    }

    @Override
    public boolean supportsNestedGroups() {
        return supportsNestedGroups;
    }

    @Override
    public boolean supportsPasswordExpiration() {
        return false;
    }

    @Override
    public boolean supportsSettingEncryptedCredential() {
        return false;
    }

    @Override
    public boolean isRolesDisabled() {
        return true;
    }

    @Nonnull
    @Override
    public Iterable<Membership> getMemberships() throws OperationFailedException {
        final Iterator<Membership> iterator = createMembershipHelper().membershipIterator();
        return () -> iterator;
    }

    @Nonnull
    @Override
    public RemoteDirectory getAuthoritativeDirectory() {
        return this;
    }

    @Override
    public void expireAllPasswords() throws OperationFailedException {
        throw new OperationNotSupportedException(NOT_IMPLEMENTED);
    }

    @Nullable
    @Override
    public AvatarReference getUserAvatarByName(final String username, final int sizeHint) {
        return null;
    }

    @Override
    public Optional<Set<String>> getLocallyFilteredGroupNames() {
        return Optional.of(getGroupsNamesToFilter())
                .filter(filter -> !filter.isEmpty() && isGroupFilteringEnabled());
    }

    public boolean isGroupFilteringEnabled() {
        return dcLicenseChecker.isDcLicense()
                && Boolean.parseBoolean(getValue(AzureAdDirectory.GROUP_FILTERING_ENABLED_ATTRIBUTE));
    }

    public Set<String> getGroupsNamesToFilter() {
        final String groupsToFilterAttribute = getValue(AzureAdDirectory.FILTERED_GROUPS_ATTRIBUTE);
        return AzureGroupFilterProcessor.getGroupNames(groupsToFilterAttribute);
    }

    public List<GroupWithAttributes> getFilteredGroups() throws OperationFailedException {
        final ImmutableList<String> groupNamesToFilter = ImmutableList.copyOf(getGroupsNamesToFilter());
        // We will return all groups when no filter is specified.
        // This is backward compatibility of unintended behaviour that we decided to keep because it doesn't break
        // anything.
        SearchRestriction searchRestriction = groupNamesToFilter.isEmpty()
                ? NullRestriction.INSTANCE
                : Restriction.on(GroupTermKeys.NAME).exactlyMatchingAny(groupNamesToFilter);
        final ImmutableList<GroupWithAttributes> groupsFromAzureAd = ImmutableList.copyOf(searchGroups(
                QueryBuilder.queryFor(GroupWithAttributes.class, EntityDescriptor.group())
                        .with(searchRestriction)
                        .returningAtMost(EntityQuery.ALL_RESULTS)));
        logNotExistingGroupNames(groupNamesToFilter, groupsFromAzureAd);
        return groupsFromAzureAd;
    }

    private void logNotExistingGroupNames(final ImmutableList<String> groupNamesToFilter, final ImmutableList<GroupWithAttributes> groupsFromAzureAd) {
        final List<String> groupNamesFromAzureAd = groupsFromAzureAd.stream().map(GroupWithAttributes::getName).collect(Collectors.toList());
        final Set<String> groupsNotFoundInAzureAd = groupNamesToFilter.stream()
                .filter(IdentifierUtils.containsIdentifierPredicate(groupNamesFromAzureAd).negate())
                .collect(Collectors.toSet());
        if (!groupsNotFoundInAzureAd.isEmpty()) {
            logger.warn("Non existent Group(s) to filter out in Azure AD: {}", groupsNotFoundInAzureAd);
        }
    }

    public DeltaQueryResult<UserWithAttributes> performUsersDeltaQuery() throws OperationFailedException {
        final ODataSelect select = graphQueryTranslator.resolveAzureAdColumnsForSingleEntityTypeQuery(EntityDescriptor.user(), FetchMode.DELTA_QUERY);
        final GraphDeltaQueryResult<GraphDeltaQueryUser> results = getAzureAdPagingWrapper().fetchAllDeltaQueryResults(getRestClient().performUsersDeltaQuery(select));
        return restEntityMapper.mapDeltaQueryUsers(results, getDirectoryId(), getAlternativeUsernameAttribute());
    }

    public DeltaQueryResult<GroupWithMembershipChanges> performGroupsDeltaQuery() throws OperationFailedException {
        final ODataExpand expand = graphQueryTranslator.resolveAzureAdNavigationPropertiesForSingleEntityTypeQuery(EntityDescriptor.group(), FetchMode.DELTA_QUERY);
        final ODataSelect select = graphQueryTranslator.resolveAzureAdColumnsForSingleEntityTypeQuery(EntityDescriptor.group(), FetchMode.DELTA_QUERY);
        final GraphDeltaQueryResult<GraphDeltaQueryGroup> results = getAzureAdPagingWrapper().fetchAllDeltaQueryResults(getRestClient().performGroupsDeltaQuery(select, expand));
        return restEntityMapper.mapDeltaQueryGroups(results, getDirectoryId());
    }

    public DeltaQueryResult<GroupWithMembershipChanges> fetchGroupChanges(final String syncToken) throws OperationFailedException {
        final GraphDeltaQueryResult<GraphDeltaQueryGroup> groups = getAzureAdPagingWrapper().fetchAllDeltaQueryResults(getRestClient().performGroupsDeltaQuery(new MicrosoftGraphDeltaToken(syncToken)));
        return restEntityMapper.mapDeltaQueryGroups(groups, getDirectoryId());
    }

    public DeltaQueryResult<UserWithAttributes> fetchUserChanges(final String syncToken) throws OperationFailedException {
        final GraphDeltaQueryResult<GraphDeltaQueryUser> users = getAzureAdPagingWrapper().fetchAllDeltaQueryResults(getRestClient().performUsersDeltaQuery(new MicrosoftGraphDeltaToken(syncToken)));
        return restEntityMapper.mapDeltaQueryUsers(users, getDirectoryId(), getAlternativeUsernameAttribute());
    }

    private Optional<String> fetchExternalIdOfGroup(final String groupName) throws OperationFailedException {
        final ODataFilter filter = graphQueryTranslator.translateSearchRestriction(
                EntityDescriptor.group(), Restriction.on(GroupTermKeys.NAME).exactlyMatching(groupName), null);
        final ODataSelect select = graphQueryTranslator.resolveAzureAdColumnsForSingleEntityTypeQuery(EntityDescriptor.group(), FetchMode.ID);
        final GraphGroupList groupList = getRestClient().searchGroups(new GraphQuery(filter, select, 0, new ODataTop(2)));
        return getOnlyElementIfPresent("group", groupName, groupList).map(GraphGroup::getId);
    }

    private Optional<String> fetchExternalIdOfUser(final String userName) throws OperationFailedException {
        try {
            return Optional.of(findUserByName(userName).getExternalId());
        } catch (UserNotFoundException e) {
            return Optional.empty();
        }
    }

    private <T> Optional<T> getOnlyElementIfPresent(String entityType, String name, PageableGraphList<T> pageableGraphList) {
        final List<T> results = pageableGraphList.getEntries();
        if (results.isEmpty()) {
            return Optional.empty();
        } else if (results.size() > 1) {
            throw new IllegalStateException(String.format("More than one %s with name %s exists", entityType, name));
        } else {
            return Optional.of(Iterables.getOnlyElement(results));
        }
    }

    public boolean supportsDeltaQueryApi() {
        return getRestClient().supportsDeltaQuery();
    }

    private <T> Predicate<DirectoryObject> getDirectoryObjectFilter(final MembershipQuery<T> query) {
        if (query.getEntityToReturn() == EntityDescriptor.user()) {
            return MembershipFilterUtil::isUser;
        } else if (query.getEntityToReturn() == EntityDescriptor.group()) {
            return MembershipFilterUtil::isGroup;
        } else {
            throw new IllegalStateException("Unsupported entity type " + query.getEntityToReturn());
        }
    }

    private <T extends Exception> void validateSingleResult(final PageableGraphList<?> results, final Supplier<T> noResultsFoundExceptionSupplier)
            throws T {
        getSingleResult(results.getEntries(), noResultsFoundExceptionSupplier);
    }

    private <T, E extends Exception> T getSingleResult(List<T> results, final Supplier<E> noResultsFoundExceptionSupplier) throws E {
        final int amountOfResults = results.size();
        if (amountOfResults == 0) {
            throw noResultsFoundExceptionSupplier.get();
        } else if (amountOfResults > 1) {
            throw new IllegalStateException(String.format("Expected one result, found %d. Please verify that there are no" +
                    "entities with duplicate names in the directory", amountOfResults));
        }
        return Iterables.getOnlyElement(results);
    }

    public AzureMembershipHelper createMembershipHelper() {
        return new AzureMembershipHelper(getRestClient(), getAzureAdPagingWrapper(), graphQueryTranslator,
                restEntityMapper, this);
    }


    @VisibleForTesting
    public AzureAdRestClient getRestClient() {
        return azureAdRestClient.get();
    }

    private UserCredentialVerifier getUserCredentialVerifier() {
        return userCredentialVerifier.get();
    }

    private AzureAdPagingWrapper getAzureAdPagingWrapper() {
        return azureAdPagingWrapper.get();
    }

    public MicrosoftGraphQueryTranslator getTranslator() {
        return graphQueryTranslator;
    }

    public boolean isLocalGroupsEnabled() {
        return localGroupsEnabled;
    }

    /**
     * Returns name of the attribute that user name should be mapped to for external users.
     */
    public String getAlternativeUsernameAttribute() {
        return dcLicenseChecker.isDcLicense() ? getValue(ALTERNATIVE_USERNAME_ATTRIBUTE) : null;
    }

    private GraphQuery convert(final EntityQuery<?> query) {
        final String alternativeAttribute = getAlternativeUsernameAttribute();
        return graphQueryTranslator.convert(query,
                StringUtils.isNotEmpty(alternativeAttribute) ? alternativeAttribute : MicrosoftGraphQueryTranslator.USERNAME);
    }
}
