package com.atlassian.crowd.model.user;

import java.util.Collections;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.base.MoreObjects;

/**
 * Mutable user template with mutable attributes.
 */
public class UserTemplateWithAttributes extends UserTemplate implements UserWithAttributes {
    private final Map<String, Set<String>> attributes = new HashMap<String, Set<String>>();

    public UserTemplateWithAttributes(String username, long directoryId) {
        super(username, directoryId);
    }

    /**
     * Creates new UserTemplateWithAttributes based on the given user with attributes.
     *
     * @param user user to use as a template
     */
    public UserTemplateWithAttributes(UserWithAttributes user) {
        super(user);

        for (String key : user.getKeys()) {
            Set<String> values = user.getValues(key);
            if (values != null) {
                this.attributes.put(key, new HashSet<String>(values));
            } else {
                throw new ConcurrentModificationException("user attributes have changed");
            }
        }
    }

    /**
     * Creates new UserTemplateWithAttributes based on the given user with empty attributes.
     *
     * @param user user to use as a template
     * @return UserTemplateWithAttributes based on the given user with empty attributes
     * @deprecated Use {@link #toUserWithNoAttributes(User)} instead. Since v2.9.
     */
    @Deprecated
    public static UserTemplateWithAttributes ofUserWithNoAttributes(User user) {
        return toUserWithNoAttributes(user);
    }

    /**
     * Creates new UserTemplateWithAttributes based on the given user with attributes.
     *
     * @param user user to use as a template
     */
    public UserTemplateWithAttributes(com.atlassian.crowd.embedded.api.UserWithAttributes user) {
        super(user);

        for (String key : user.getKeys()) {
            this.attributes.put(key, new HashSet<String>(user.getValues(key)));
        }
    }

    protected UserTemplateWithAttributes(User user) {
        super(user);
    }

    protected UserTemplateWithAttributes(com.atlassian.crowd.embedded.api.User user) {
        super(user);
    }

    /**
     * Creates new UserTemplateWithAttributes based on the given user with empty attributes.
     *
     * @param user user to use as a template
     * @return UserTemplateWithAttributes based on the given user with empty attributes
     */
    public static UserTemplateWithAttributes toUserWithNoAttributes(User user) {
        return new UserTemplateWithAttributes(user);
    }

    /**
     * Creates new UserTemplateWithAttributes based on the given user with empty attributes.
     *
     * @param user user to use as a template
     * @return UserTemplateWithAttributes based on the given user with empty attributes
     */
    public static UserTemplateWithAttributes toUserWithNoAttributes(com.atlassian.crowd.embedded.api.User user) {
        return new UserTemplateWithAttributes(user);
    }

    public Map<String, Set<String>> getAttributes() {
        return attributes;
    }

    @Override
    @Nullable
    public Set<String> getValues(String name) {
        return attributes.get(name);
    }

    @Override
    @Nullable
    public String getValue(String name) {
        Set<String> values = getValues(name);
        if (values != null && !values.isEmpty()) {
            return values.iterator().next();
        } else {
            return null;
        }
    }

    @Override
    public Set<String> getKeys() {
        return attributes.keySet();
    }

    @Override
    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    public void setAttribute(String name, String value) {
        attributes.put(name, Collections.singleton(value));
    }

    public void setAttribute(String name, Set<String> values) {
        attributes.put(name, values);
    }

    public void removeAttribute(String name) {
        attributes.remove(name);
    }

    public String toString() {
        return MoreObjects.toStringHelper(this).
                add("name", getName()).
                add("directoryId", getDirectoryId()).
                add("active", isActive()).
                add("emailAddress", getEmailAddress()).
                add("firstName", getFirstName()).
                add("lastName", getLastName()).
                add("displayName", getDisplayName()).
                add("externalId", getExternalId()).
                add("attributes", getAttributes()).
                toString();
    }

}
