package com.atlassian.crowd.manager.application;

import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.application.ApplicationDirectoryMapping;

import java.util.List;

/**
 * The service for configuring default group memberships.
 *
 * @since v3.1.0
 */
public interface DefaultGroupMembershipService {

    /**
     * Adds a new default group membership for the given application and group.
     * @param application the application for which the default group membership will be created
     * @param directoryMapping directory mapping between the application and the directory from which the group originates,
     *                         must be added to the application before calling this method
     * @param groupName the name of the group in the default group membership configuration
     * @throws OperationFailedException when the directory mapping does not exist
     */
    void add(Application application, ApplicationDirectoryMapping directoryMapping, String groupName) throws OperationFailedException;

    /**
     * Removes a default group membership configuration for the given application and group.
     * @param application the application for which the default group membership will be removed
     * @param directoryMapping directory mapping between the application and the directory from which the group originates,
     *                         must be added to the application before calling this method
     * @param groupName the name of the group in the default group membership configuration
     * @throws OperationFailedException when the directory mapping does not exist
     */
    void remove(Application application, ApplicationDirectoryMapping directoryMapping, String groupName) throws OperationFailedException;

    /**
     * Lists all default group membership configurations for the given application/directory combination
     * @param application the application for which the default group memberships should be listed
     * @param directoryMapping directory mapping between the application and the directory from which the group originates,
     *                         must be added to the application before calling this method
     * @throws OperationFailedException when the directory mapping does not exist
     * @return a list of group names representing the default group memberships for the given application/directory
     * combination
     */
    List<String> listAll(Application application, ApplicationDirectoryMapping directoryMapping) throws OperationFailedException;

}
