package com.atlassian.crowd.audit.query;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.crowd.audit.AuditLogChangeset;
import com.atlassian.crowd.audit.AuditLogEventSource;
import com.atlassian.crowd.audit.AuditLogEventType;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;

import java.time.Instant;
import java.util.Collection;

/**
 * Allows creating AuditLogQueries
 * @see AuditLogQuery
 * @since v3.2
 */
@ExperimentalApi
public class AuditLogQueryBuilder<RESULT> {
    Class<RESULT> resultClass;
    Instant beforeOrOn;
    Instant onOrAfter;
    ImmutableList.Builder<AuditLogEventType> actions = new ImmutableList.Builder<>();
    ImmutableList.Builder<AuditLogEventSource> sources = new ImmutableList.Builder<>();
    ImmutableList.Builder<AuditLogQueryAuthorRestriction> authors = new ImmutableList.Builder<>();
    ImmutableList.Builder<AuditLogQueryEntityRestriction> users = new ImmutableList.Builder<>();
    ImmutableList.Builder<AuditLogQueryEntityRestriction> groups = new ImmutableList.Builder<>();
    ImmutableList.Builder<AuditLogQueryEntityRestriction> applications = new ImmutableList.Builder<>();
    ImmutableList.Builder<AuditLogQueryEntityRestriction> directories = new ImmutableList.Builder<>();
    AuditLogChangesetProjection projection;

    int startIndex;
    int maxResults = EntityQuery.MAX_MAX_RESULTS;

    /**
     * @return a builder for queries returning changesets - this is the most common use for audit log queries
     */
    public static AuditLogQueryBuilder<AuditLogChangeset> changesetQuery() {
        return queryFor(AuditLogChangeset.class);
    }

    /**
     * @return a builder for queries returning a specific types of results - this is mostly used with changeset
     * projections, for example to return matching authors or entities.
     * Available result types might be different for different queries.
     * @see AuditLogChangesetProjection
     */
    public static <RESULT> AuditLogQueryBuilder<RESULT> queryFor(Class<RESULT> result) {
        return new AuditLogQueryBuilder<>(result);
    }

    private AuditLogQueryBuilder(Class<RESULT> resultClass) {
        this.resultClass = resultClass;
    }

    public AuditLogQueryBuilder<RESULT> setBeforeOrOn(Instant beforeOrOn) {
        this.beforeOrOn = beforeOrOn;
        return this;
    }

    public AuditLogQueryBuilder<RESULT> setOnOrAfter(Instant onOrAfter) {
        this.onOrAfter = onOrAfter;
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addAllActions(Collection<AuditLogEventType> actions) {
        this.actions.addAll(actions);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addAction(AuditLogEventType action) {
        this.actions.add(action);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addAllSources(Collection<AuditLogEventSource> sources) {
        this.sources.addAll(sources);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addSource(AuditLogEventSource source) {
        this.sources.add(source);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addAllAuthors(Collection<AuditLogQueryAuthorRestriction> authors) {
        this.authors.addAll(authors);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addAuthor(AuditLogQueryAuthorRestriction authors) {
        this.authors.add(authors);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addAllUsers(Collection<AuditLogQueryEntityRestriction> user) {
        this.users.addAll(user);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addUser(AuditLogQueryEntityRestriction user) {
        this.users.add(user);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> setUsers(Collection<AuditLogQueryEntityRestriction> users) {
        this.users = ImmutableList.<AuditLogQueryEntityRestriction>builder().addAll(users);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addAllGroups(Collection<AuditLogQueryEntityRestriction> groups) {
        this.groups.addAll(groups);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addGroup(AuditLogQueryEntityRestriction group) {
        this.groups.add(group);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> setGroups(Collection<AuditLogQueryEntityRestriction> groups) {
        this.groups = ImmutableList.<AuditLogQueryEntityRestriction>builder().addAll(groups);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addAllApplications(Collection<AuditLogQueryEntityRestriction> applications) {
        this.applications.addAll(applications);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addApplication(AuditLogQueryEntityRestriction application) {
        this.applications.add(application);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> setApplications(Collection<AuditLogQueryEntityRestriction> applications) {
        this.applications = ImmutableList.<AuditLogQueryEntityRestriction>builder().addAll(applications);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addAllDirectories(Collection<AuditLogQueryEntityRestriction> directories) {
        this.directories.addAll(directories);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> addDirectory(AuditLogQueryEntityRestriction directory) {
        this.directories.add(directory);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> setDirectories(Collection<AuditLogQueryEntityRestriction> directories) {
        this.directories = ImmutableList.<AuditLogQueryEntityRestriction>builder().addAll(directories);
        return this;
    }

    public AuditLogQueryBuilder<RESULT> setProjection(AuditLogChangesetProjection projection) {
        Preconditions.checkState(this.projection == null, "Query projection already set to " + this.projection);
        this.projection = projection;
        return this;
    }

    public AuditLogQueryBuilder<RESULT> setStartIndex(int startIndex) {
        this.startIndex = startIndex;
        return this;
    }

    public AuditLogQueryBuilder<RESULT> setMaxResults(int maxResults) {
        this.maxResults = maxResults;
        return this;
    }

    public AuditLogQuery<RESULT> build() {
        return new AuditLogQuery<>(this);
    }
}