package com.atlassian.crowd.util;

import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonProperty;
import org.slf4j.Logger;

import java.util.Objects;
import java.util.Optional;

import static org.slf4j.LoggerFactory.getLogger;

/**
 * Class represents information about image based on it's data URL
 */
public class ImageInfo {
    private static final Logger log = getLogger(ImageInfo.class);
    private static final String IMAGE_FILE_PREFIX = "image-";
    private static final String SEPARATOR = ",";
    private static final String DATA_IMAGE_PREFIX = "data:image/";
    private static final String BASE_64_IMAGE_PREFIX = ";base64";

    private final String name;
    private final String imageBase64;
    private final String extension;

    @JsonCreator
    public ImageInfo(@JsonProperty("name") String name,
                     @JsonProperty("imageBase64") String imageBase64,
                     @JsonProperty("extension") String extension) {
        this.name = name;
        this.imageBase64 = imageBase64;
        this.extension = extension;
    }

    /**
     * Extracts base64 encoded image data to the {@link ImageInfo}
     * @param imageData base64 encoded image (e.g. data:image/png;base64, iVBORw0KG...==)
     * @return Image info or empty
     */
    public static Optional<ImageInfo> fromImageData(String imageData) {
        try {
            final String clearedImage = imageData.trim()
                    .replace(DATA_IMAGE_PREFIX, "")
                    .replace(BASE_64_IMAGE_PREFIX, "");
            final int commaIndex = clearedImage.indexOf(SEPARATOR);

            final String extension = clearedImage.substring(0, commaIndex);
            final String imageBase64 = clearedImage.substring(commaIndex + 1);
            final String imageName = IMAGE_FILE_PREFIX + Math.abs(Objects.hash(imageBase64));

            return Optional.of(new ImageInfo(imageName, imageBase64, extension));
        } catch (Exception e) {
            log.debug("Error while extracting imageData {}", imageData, e);
            return Optional.empty();
        }
    }

    public String getName() {
        return name;
    }

    public String getImageBase64() {
        return imageBase64;
    }

    public String getExtension() {
        return extension;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ImageInfo imageInfo = (ImageInfo) o;
        return Objects.equals(name, imageInfo.name) &&
                Objects.equals(imageBase64, imageInfo.imageBase64) &&
                Objects.equals(extension, imageInfo.extension);
    }

    @Override
    public int hashCode() {
        return Objects.hash(name, imageBase64, extension);
    }
}
