package com.atlassian.crowd.model.webhook;

import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.application.ImmutableApplication;
import com.google.common.base.MoreObjects;

import java.util.Date;
import java.util.Objects;

public class ImmutableWebhook implements Webhook {
    private final Long id;
    private final String endpointUrl;
    private final ImmutableApplication application;
    private final String token;
    private final Date oldestFailureDate;
    private final long failuresSinceLastSuccess;

    protected ImmutableWebhook(Long id, String endpointUrl, ImmutableApplication application, String token, Date oldestFailureDate, long failuresSinceLastSuccess) {
        this.id = id;
        this.endpointUrl = endpointUrl;
        this.application = application;
        this.token = token;
        this.oldestFailureDate = oldestFailureDate;
        this.failuresSinceLastSuccess = failuresSinceLastSuccess;
    }

    public Long getId() {
        return id;
    }

    public String getEndpointUrl() {
        return endpointUrl;
    }

    public Application getApplication() {
        return application;
    }

    public String getToken() {
        return token;
    }

    public Date getOldestFailureDate() {
        return oldestFailureDate;
    }

    public long getFailuresSinceLastSuccess() {
        return failuresSinceLastSuccess;
    }

    public static ImmutableWebhook.Builder builder() {
        return new ImmutableWebhook.Builder();
    }

    public static ImmutableWebhook.Builder builder(Webhook data) {
        return new ImmutableWebhook.Builder(data);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        ImmutableWebhook that = (ImmutableWebhook) o;

        return Objects.equals(this.getId(), that.getId()) &&
                Objects.equals(this.getEndpointUrl(), that.getEndpointUrl()) &&
                Objects.equals(this.getApplication(), that.getApplication()) &&
                Objects.equals(this.getToken(), that.getToken()) &&
                Objects.equals(this.getOldestFailureDate(), that.getOldestFailureDate()) &&
                Objects.equals(this.getFailuresSinceLastSuccess(), that.getFailuresSinceLastSuccess());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getId(), getEndpointUrl(), getApplication(), getToken(), getOldestFailureDate(), getFailuresSinceLastSuccess());
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("id", getId())
                .add("endpointUrl", getEndpointUrl())
                .add("application", getApplication())
                .add("token", getToken())
                .add("oldestFailureDate", getOldestFailureDate())
                .add("failuresSinceLastSuccess", getFailuresSinceLastSuccess())
                .toString();
    }

    public static final class Builder {

        private Long id;
        private String endpointUrl;
        private ImmutableApplication application;
        private String token;
        private Date oldestFailureDate;
        private long failuresSinceLastSuccess;

        private Builder() {
        }

        private Builder(Webhook webhook) {
            this.id = webhook.getId();
            this.endpointUrl = webhook.getEndpointUrl();
            this.token = webhook.getToken();
            this.oldestFailureDate = webhook.getOldestFailureDate();
            this.failuresSinceLastSuccess = webhook.getFailuresSinceLastSuccess();
        }

        public Builder setId(Long id) {
            this.id = id;
            return this;
        }


        public Builder setEndpointUrl(String endpointUrl) {
            this.endpointUrl = endpointUrl;
            return this;
        }


        public Builder setApplication(ImmutableApplication application) {
            this.application = application;
            return this;
        }


        public Builder setToken(String token) {
            this.token = token;
            return this;
        }


        public Builder setOldestFailureDate(Date oldestFailureDate) {
            this.oldestFailureDate = oldestFailureDate;
            return this;
        }


        public Builder setFailuresSinceLastSuccess(long failuresSinceLastSuccess) {
            this.failuresSinceLastSuccess = failuresSinceLastSuccess;
            return this;
        }


        public ImmutableWebhook build() {
            return new ImmutableWebhook(id, endpointUrl, application, token, oldestFailureDate, failuresSinceLastSuccess);
        }
    }
}
