package com.atlassian.crowd.model.directory;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.DirectoryType;
import com.atlassian.crowd.embedded.api.OperationType;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;

import javax.annotation.Nullable;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

public class ImmutableDirectory implements Directory {
    private final Long id;
    private final String name;
    private final boolean active;
    private final String encryptionType;
    private final Map<String, String> attributes;
    private final Set<OperationType> allowedOperations;
    private final String description;
    private final DirectoryType type;
    private final String implementationClass;
    private final Date createdDate;
    private final Date updatedDate;

    private ImmutableDirectory(final Builder builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.active = builder.active;
        this.encryptionType = builder.encryptionType;
        this.attributes = Collections.unmodifiableMap(new HashMap<>(builder.attributes));
        this.allowedOperations = Collections.unmodifiableSet(new HashSet<>(builder.allowedOperations));
        this.description = builder.description;
        this.type = builder.type;
        this.implementationClass = builder.implementationClass;
        this.createdDate = builder.createdDate;
        this.updatedDate = builder.updatedDate;
    }

    @Override
    public Long getId() {
        return id;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public String getEncryptionType() {
        return encryptionType;
    }

    @Override
    public Map<String, String> getAttributes() {
        return attributes;
    }

    @Override
    public Set<OperationType> getAllowedOperations() {
        return allowedOperations;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public DirectoryType getType() {
        return type;
    }

    @Override
    public String getImplementationClass() {
        return implementationClass;
    }

    @Override
    public Date getCreatedDate() {
        return createdDate == null ? null : new Date(createdDate.getTime());
    }

    @Override
    public Date getUpdatedDate() {
        return updatedDate == null ? null : new Date(updatedDate.getTime());
    }

    @Nullable
    @Override
    public Set<String> getValues(String key) {
        final String value = attributes.get(key);
        return value == null ? null : Collections.singleton(value);
    }

    @Nullable
    @Override
    public String getValue(String key) {
        return attributes.get(key);
    }

    @Override
    public Set<String> getKeys() {
        return attributes.keySet();
    }

    @Override
    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ImmutableDirectory that = (ImmutableDirectory) o;
        return active == that.active &&
                Objects.equals(id, that.id) &&
                Objects.equals(name, that.name) &&
                Objects.equals(encryptionType, that.encryptionType) &&
                Objects.equals(attributes, that.attributes) &&
                Objects.equals(allowedOperations, that.allowedOperations) &&
                Objects.equals(description, that.description) &&
                type == that.type &&
                Objects.equals(implementationClass, that.implementationClass) &&
                Objects.equals(createdDate, that.createdDate) &&
                Objects.equals(updatedDate, that.updatedDate);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, name, active, encryptionType, attributes, allowedOperations, description, type, implementationClass, createdDate, updatedDate);
    }

    public static ImmutableDirectory from(Directory directory) {
        return directory instanceof ImmutableDirectory ? (ImmutableDirectory) directory : builder(directory).build();
    }

    public static Builder builder(String name, DirectoryType type, String implementationClass) {
        return new Builder(name, type, implementationClass);
    }

    public static Builder builder(Directory directory) {
        return new Builder(directory);
    }

    public static class Builder {
        private Long id;
        private String name;
        private boolean active = true;
        private String encryptionType;
        private Map<String, String> attributes = new HashMap<>();
        private Set<OperationType> allowedOperations = new HashSet<>();
        private String description;
        private DirectoryType type;
        private String implementationClass;
        private Date createdDate;
        private Date updatedDate;

        public Builder(String name, DirectoryType type, String implementationClass) {
            this.name = name;
            this.type = type;
            this.implementationClass = implementationClass;
        }

        public Builder(final Directory directory) {
            Preconditions.checkNotNull(directory, "directory");
            this.setName(directory.getName())
                    .setType(directory.getType())
                    .setImplementationClass(directory.getImplementationClass())
                    .setId(directory.getId())
                    .setActive(directory.isActive())
                    .setEncryptionType(directory.getEncryptionType())
                    .setAttributes(new HashMap<>(directory.getAttributes()))
                    .setAllowedOperations(directory.getAllowedOperations())
                    .setDescription(directory.getDescription())
                    .setCreatedDate(directory.getCreatedDate())
                    .setUpdatedDate(directory.getUpdatedDate());
        }

        public Builder setId(Long id) {
            this.id = id;
            return this;
        }

        public Builder setName(String name) {
            this.name = name;
            return this;
        }

        public Builder setActive(boolean active) {
            this.active = active;
            return this;
        }

        public Builder setEncryptionType(String encryptionType) {
            this.encryptionType = encryptionType;
            return this;
        }

        public Builder setAttributes(Map<String, String> attributes) {
            this.attributes = new HashMap<>(attributes);
            return this;
        }

        public Builder setAttribute(String key, String value) {
            this.attributes.put(key, value);
            return this;
        }

        public Builder setAllowedOperations(Set<OperationType> allowedOperations) {
            this.allowedOperations = allowedOperations;
            return this;
        }

        public Builder setDescription(String description) {
            this.description = description;
            return this;
        }

        public Builder setType(DirectoryType type) {
            this.type = type;
            return this;
        }

        public Builder setImplementationClass(String implementationClass) {
            this.implementationClass = implementationClass;
            return this;
        }

        public Builder setCreatedDate(Date createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public Builder setUpdatedDate(Date updatedDate) {
            this.updatedDate = updatedDate;
            return this;
        }

        public ImmutableDirectory build() {
            return new ImmutableDirectory(this);
        }
    }

    @Override
    public String toString() {
        // intentionally omitting attributes to avoid exposing sensitive ones
        return MoreObjects.toStringHelper(this)
                .add("id", id)
                .add("name", name)
                .add("type", type)
                .toString();
    }
}
