package com.atlassian.crowd.manager.property;

import com.atlassian.crowd.exception.ObjectNotFoundException;
import com.atlassian.crowd.manager.audit.AuditLogConfiguration;
import com.atlassian.crowd.model.authentication.CookieConfiguration;
import com.atlassian.crowd.model.backup.BackupConfiguration;
import com.atlassian.crowd.model.lookandfeel.LookAndFeelConfiguration;
import com.atlassian.crowd.util.ImageInfo;
import com.atlassian.crowd.util.mail.SMTPServer;

import java.net.URI;
import java.security.Key;
import java.util.Optional;

/**
 * API for storing and retrieving Crowd server properties.
 */
public interface PropertyManager {
    /**
     * @return deployment title.
     * @throws PropertyManagerException property does not exist.
     */
    String getDeploymentTitle() throws PropertyManagerException;

    /**
     * @param title deployment title.
     */
    void setDeploymentTitle(String title);

    /**
     * Will return the Domain property from the datastore or null if the domain has
     * not been set.
     *
     * @return domain or null
     * @deprecated Use {@link #getCookieConfiguration()} instead. Since v3.0.
     */
    @Deprecated
    String getDomain();

    /**
     * @param domain SSO cookie domain.
     * @deprecated Use {@link #setCookieConfiguration(CookieConfiguration)} instead. Since v3.0.
     */
    @Deprecated
    void setDomain(String domain);

    /**
     * @return {@code true} if the "secure" flag should be set on the SSO cookie.
     * @deprecated Use {@link #setCookieConfiguration(CookieConfiguration)} instead. Since v3.0.
     */
    @Deprecated
    boolean isSecureCookie();

    /**
     * @param secure {@code true} if the "secure" flag should be set on the SSO cookie.
     * @deprecated Use {@link #setCookieConfiguration(CookieConfiguration)} instead. Since v3.0.
     */
    @Deprecated
    void setSecureCookie(boolean secure);

    /**
     * @param enabled {@code true} if application authorisation caching should be used on the server-side.
     */
    void setCacheEnabled(boolean enabled);

    /**
     * @return {@code true} if application authorisation caching is used on the server-side.
     */
    boolean isCacheEnabled();

    /**
     * @return number of minutes the session is valid.
     */
    long getSessionTime();

    /**
     * @param time number of minutes the session is valid.
     */
    void setSessionTime(long time);

    /**
     * @return SMTP server config.
     * @throws PropertyManagerException property does not exist.
     * @deprecated As of release 3.3.0, use {@link com.atlassian.crowd.manager.mail.MailConfigurationService#getConfiguration()}
     */
    @Deprecated
    SMTPServer getSMTPServer() throws PropertyManagerException;

    /**
     * @param server SMTP server config.
     * @deprecated As of release 3.3.0, use {@link com.atlassian.crowd.manager.mail.MailConfigurationService#saveConfiguration()}
     */
    @Deprecated
    void setSMTPServer(SMTPServer server);

    /**
     * @return DES key for DES encoded passwords.
     * @throws PropertyManagerException property does not exist.
     */
    Key getDesEncryptionKey() throws PropertyManagerException;

    /**
     * Generates and stores a DES key for DES encoded passwords.
     *
     * @throws PropertyManagerException DES algorithm does not exist.
     */
    void generateDesEncryptionKey() throws PropertyManagerException;

    /**
     * @param template mail template.
     * @deprecated As of release 2.1, use {@link #setProperty(String, String)}
     */
    @Deprecated
    void setSMTPTemplate(String template);

    /**
     * @return mail template.
     * @throws PropertyManagerException property does not exist.
     * @deprecated As of release 2.1, use {@link #getProperty(String)}
     */
    @Deprecated
    String getSMTPTemplate() throws PropertyManagerException;

    /**
     * @param total license resource total.
     */
    void setCurrentLicenseResourceTotal(int total);

    /**
     * @return license resource total.
     */
    int getCurrentLicenseResourceTotal();

    /**
     * @param notificationEmail notification email.
     * @deprecated As of release 3.3, use {@link com.atlassian.crowd.manager.mail.MailConfigurationService#saveConfiguration()}
     */
    @Deprecated
    void setNotificationEmail(String notificationEmail);

    /**
     * @return notification email.
     * @throws PropertyManagerException property does not exist
     * @deprecated As of release 3.3, use {@link com.atlassian.crowd.manager.mail.MailConfigurationService#getMailConfiguration()}
     */
    @Deprecated
    String getNotificationEmail() throws PropertyManagerException;

    /**
     * @return {@code true} if GZip compression should be used.
     * @throws PropertyManagerException property does not exist.
     * @deprecated this is no longer configurable in Crowd and always returns <code>true</code>
     */
    @Deprecated
    boolean isGzipEnabled() throws PropertyManagerException;

    /**
     * @param gzip {@code true} if GZip compression should be used.
     * @deprecated this is no longer configurable in Crowd and this method has no effect.
     */
    @Deprecated
    void setGzipEnabled(boolean gzip);

    /**
     * This method returns the current build number for Crowd from the datastore. This BuildNumber may not be the same
     * as the build number in {@link com.atlassian.crowd.util.build.BuildUtils#BUILD_NUMBER} since this number is for the
     * current release of Crowd, while the number in the database may still be set to a previous version if the UpgradeManager
     * has not been run.
     *
     * @return an Integer representing the current build number in the database.
     * @throws PropertyManagerException if we fail to find the buildNumber
     */
    Integer getBuildNumber() throws PropertyManagerException;

    /**
     * Will set the buildNumber for the current release of Crowd.
     *
     * @param buildNumber the buildNumber to set in the database
     */
    void setBuildNumber(Integer buildNumber);

    /**
     * Retrieves a String that contains a list of proxy servers we trust to correctly set the X-Forwarded-For flag.
     * Internal format of this string is the responsibility of TrustedProxyManagerImpl.
     *
     * @return list of proxy servers as a string.
     * @throws PropertyManagerException If the list of proxy servers could not be found.
     */
    String getTrustedProxyServers() throws PropertyManagerException;

    /**
     * Persists a String containing a list of proxy servers we trust to correctly set the X-Forwarded-For flag.
     * Internal format of this string is the responsibility of TrustedProxyManagerImpl.
     *
     * @param proxyServers proxy servers.
     */
    void setTrustedProxyServers(String proxyServers);

    /**
     * Persists the audit log configuration
     *
     * @param auditLogConfiguration new configuration
     */
    void setAuditLogConfiguration(AuditLogConfiguration auditLogConfiguration);

    /**
     * Retrieves the audit log configuration
     *
     * @return audit log configuration, default if not found
     */
    AuditLogConfiguration getAuditLogConfiguration();

    /**
     * Will return true if the Crowd instance is using database token storage for authentication {@link com.atlassian.crowd.model.token.Token}'s
     * otherwise assume we are using in-memory
     *
     * @return true if database token storage is being used.
     * @throws PropertyManagerException property does not exist.
     */
    boolean isUsingDatabaseTokenStorage() throws PropertyManagerException;

    /**
     * Will set a property to state that this crowd instance is using database token storage, otherwise assume we are using in-memory
     *
     * @param usingDatabaseTokenStorage true if you are switching to in-memory token storage
     */
    void setUsingDatabaseTokenStorage(boolean usingDatabaseTokenStorage);

    /**
     * Will attempt to remove a property from the datastore
     *
     * @param name the name of the property.
     */
    void removeProperty(String name);

    /**
     * Retrieves an arbitrary property by name.
     *
     * @param name name of property.
     * @return value.
     * @throws ObjectNotFoundException property does not exist.
     */
    String getProperty(String name) throws ObjectNotFoundException;

    /**
     * Retrieves an arbitrary property by name.
     *
     * @param name name of property.
     * @return value if found, none otherwise
     */
    Optional<String> getOptionalProperty(String name);

    /**
     * Sets an arbitrary property.
     *
     * @param name  name of property.
     * @param value value.
     */
    void setProperty(String name, String value);

    /**
     * Should the client IP address be included as a validation factor?
     */
    boolean isIncludeIpAddressInValidationFactors();

    void setIncludeIpAddressInValidationFactors(boolean includeIpAddressInValidationFactors);

    /**
     * Should public services be used for user avatars?
     */
    boolean isUseWebAvatars();

    void setUseWebAvatars(boolean useWebAvatars);

    /**
     * @return the current configuration of Crowd's cookie-based SSO
     * @since v3.0
     */
    CookieConfiguration getCookieConfiguration();

    /**
     * Updates the configuration of Crowd's cookie-based SSO
     * @param cookieConfiguration the configuration to set, will replace the existing configuration
     * @since v3.0
     */
    void setCookieConfiguration(CookieConfiguration cookieConfiguration);

    /**
     * Gets a string property. If the property is not defined, return the supplied default value.
     *
     * @param property     a property name
     * @param defaultValue the default value to use if the property does not exist
     * @return the value of the property, or defaultValue if the property didn't exist
     */
    String getString(String property, String defaultValue);

    /**
     * Gets a boolean property. If the property is not defined, return the supplied default value.
     *
     * @param property     a property name
     * @param defaultValue the default value to use if the property does not exist
     * @return the value of the property, or defaultValue if the property didn't exist
     */
    boolean getBoolean(String property, boolean defaultValue);

    /**
     * Gets an integer property. If the property is not defined, or does not contain a valid integer, return the supplied default value.
     *
     * @param property     a property name
     * @param defaultValue the default value to use if the property does not exist or is invalid
     * @return the value of the property, or defaultValue if the property didn't exist or was corrupted
     */
    int getInt(String property, int defaultValue);

    /**
     * Sets the base URL for this Crowd application instance
     * @param url the base url to set, should be a proper absolute URL
     */
    void setBaseUrl(URI url);

    /**
     * @return the current base url for this Crowd application instance
     * @throws PropertyManagerException if the property is not configured correctly
     * @see com.atlassian.sal.api.ApplicationProperties#getBaseUrl(com.atlassian.sal.api.UrlMode)
     */
    URI getBaseUrl() throws PropertyManagerException;

    /**
     * @return ID of private key/certificate used to sign assertions in SAML if found, none otherwise
     */
    Optional<Long> getPrivateKeyCertificatePairToSign();

    /**
     * Sets ID of private key/certificate pair used to sign assertions in SAML
     * @param privateKeyCertificatePairId ID of private key/certificate pair to save
     */
    void setPrivateKeyCertificateToSign(long privateKeyCertificatePairId);

    /**
     * @return Backup configuration
     */
    BackupConfiguration getBackupConfiguration();

    /**
     * Saves backup configuration
     * @param config configuration of backup to save
     */
    void saveBackupConfiguration(BackupConfiguration config);

    /**
     * Sets Look and Feel configuration
     * @param lookAndFeelConfiguration look and feel configuration
     * @param updatedLogoInfo information about custom logo. If custom logo was already set
     *                        and new value is not null then old logo will be overridden by new one.
     * @throws PropertyManagerException if there was an error while saving configuration
     */
    void setLookAndFeelConfiguration(LookAndFeelConfiguration lookAndFeelConfiguration, ImageInfo updatedLogoInfo) throws PropertyManagerException;

    /**
     * @return Look and Feel configuration if configured, none otherwise
     * @throws PropertyManagerException if there was an error while fetching configuration
     */
    Optional<LookAndFeelConfiguration> getLookAndFeelConfiguration() throws PropertyManagerException;

    /**
     * Removes Look and Feel configuration
     * @throws PropertyManagerException if there was an error while removing configuration
     */
    void removeLookAndFeelConfiguration() throws PropertyManagerException;

    /**
     * @return logo image if configured, none otherwise
     * @throws PropertyManagerException if there was an error while fetching a logo
     */
    Optional<ImageInfo> getLogoImage() throws PropertyManagerException;
}
