package com.atlassian.crowd.audit.query;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.crowd.audit.AuditLogEventSource;
import com.atlassian.crowd.audit.AuditLogEventType;
import com.google.common.base.MoreObjects;

import javax.annotation.Nullable;
import java.time.Instant;
import java.util.List;
import java.util.Objects;

/**
 * A specialized type of query for the audit log. The restrictions will be applied as follows:
 * <ul>
 *     <li>The top-level query is a conjunction of the specified restrictions - matching changesets must have a matching
 *     start date and matching and date and matching actions and so on</li>
 *     <li>Actions, author and entity restrictions are disjunctions - matching changesets must match at least one restriction specified</li>
 *     <li>
 *         Every author/entity restriction is a conjunction of its criteria - if a restriction specifies an id, type and name,
 *         all of them will have to match
 *     </li>
 * </ul>
 *
 * The query will be ordered by the changeset timestamp, descending.
 */
@ExperimentalApi
public class AuditLogQuery<RESULT> {
    private final Instant beforeOrOn;
    private final Instant onOrAfter;
    private final List<AuditLogEventType> actions;
    private final List<AuditLogEventSource> sources;
    private final List<AuditLogQueryAuthorRestriction> authors;
    private final List<AuditLogQueryEntityRestriction> users;
    private final List<AuditLogQueryEntityRestriction> groups;
    private final List<AuditLogQueryEntityRestriction> applications;
    private final List<AuditLogQueryEntityRestriction> directories;

    private final AuditLogChangesetProjection projection;

    private final int startIndex;
    private final int maxResults;
    private final Class<RESULT> resultClass;

    AuditLogQuery(AuditLogQueryBuilder<RESULT> builder) {
        this.beforeOrOn = builder.beforeOrOn;
        this.onOrAfter = builder.onOrAfter;
        this.actions = builder.actions.build();
        this.sources = builder.sources.build();
        this.authors = builder.authors.build();
        this.users = builder.users.build();
        this.groups = builder.groups.build();
        this.applications = builder.applications.build();
        this.directories = builder.directories.build();
        this.startIndex = builder.startIndex;
        this.maxResults = builder.maxResults;
        this.projection = builder.projection;
        this.resultClass = builder.resultClass;
    }

    public Instant getBeforeOrOn() {
        return beforeOrOn;
    }

    public Instant getOnOrAfter() {
        return onOrAfter;
    }

    public List<AuditLogEventType> getActions() {
        return actions;
    }

    public List<AuditLogEventSource> getSources() {
        return sources;
    }

    public List<AuditLogQueryAuthorRestriction> getAuthors() {
        return authors;
    }

    public List<AuditLogQueryEntityRestriction> getUsers() {
        return users;
    }

    public List<AuditLogQueryEntityRestriction> getGroups() {
        return groups;
    }

    public List<AuditLogQueryEntityRestriction> getApplications() {
        return applications;
    }

    public List<AuditLogQueryEntityRestriction> getDirectories() {
        return directories;
    }

    public int getStartIndex() {
        return startIndex;
    }

    public int getMaxResults() {
        return maxResults;
    }

    @Nullable
    public AuditLogChangesetProjection getProjection() {
        return projection;
    }

    public Class<RESULT> getReturnType() {
        return resultClass;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AuditLogQuery<?> that = (AuditLogQuery<?>) o;
        return startIndex == that.startIndex &&
                maxResults == that.maxResults &&
                Objects.equals(beforeOrOn, that.beforeOrOn) &&
                Objects.equals(onOrAfter, that.onOrAfter) &&
                Objects.equals(actions, that.actions) &&
                Objects.equals(sources, that.sources) &&
                Objects.equals(authors, that.authors) &&
                Objects.equals(users, that.users) &&
                Objects.equals(groups, that.groups) &&
                Objects.equals(applications, that.applications) &&
                Objects.equals(directories, that.directories) &&
                projection == that.projection &&
                Objects.equals(resultClass, that.resultClass);
    }

    @Override
    public int hashCode() {
        return Objects.hash(beforeOrOn, onOrAfter, actions, sources, authors, users, groups, applications, directories, projection, startIndex, maxResults, resultClass);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("beforeOrOn", beforeOrOn)
                .add("onOrAfter", onOrAfter)
                .add("actions", actions)
                .add("sources", sources)
                .add("authors", authors)
                .add("users", users)
                .add("groups", groups)
                .add("applications", applications)
                .add("directories", directories)
                .add("projection", projection)
                .add("startIndex", startIndex)
                .add("maxResults", maxResults)
                .add("resultClass", resultClass)
                .toString();
    }
}
