package com.atlassian.crowd.audit;

import com.google.common.base.MoreObjects;

import java.util.Objects;

/**
 * Concrete implementation of an audit log entry
 */
public class ImmutableAuditLogEntry implements AuditLogEntry {
    private final String propertyName;
    private final String oldValue;
    private final String newValue;

    public ImmutableAuditLogEntry(String propertyName, String oldValue, String newValue) {
        this.propertyName = propertyName;
        this.oldValue = oldValue;
        this.newValue = newValue;
    }

    @Override
    public String getPropertyName() {
        return propertyName;
    }

    @Override
    public String getOldValue() {
        return oldValue;
    }

    @Override
    public String getNewValue() {
        return newValue;
    }

    public static ImmutableAuditLogEntry from(AuditLogEntry auditLogEntry) {
        return auditLogEntry instanceof ImmutableAuditLogEntry
                ? (ImmutableAuditLogEntry) auditLogEntry : new Builder(auditLogEntry).build();
    }

    public static Builder builder() {
        return new Builder();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ImmutableAuditLogEntry that = (ImmutableAuditLogEntry) o;
        return Objects.equals(propertyName, that.propertyName) &&
                Objects.equals(oldValue, that.oldValue) &&
                Objects.equals(newValue, that.newValue);
    }

    @Override
    public int hashCode() {
        return Objects.hash(propertyName, oldValue, newValue);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("propertyName", propertyName)
                .add("oldValue", oldValue)
                .add("newValue", newValue)
                .toString();
    }

    public static class Builder {
        private String propertyName;
        private String oldValue;
        private String newValue;

        public Builder() {
        }

        public Builder(AuditLogEntry entry) {
            this.propertyName = entry.getPropertyName();
            this.oldValue = entry.getOldValue();
            this.newValue = entry.getNewValue();
        }

        public Builder setPropertyName(String propertyName) {
            this.propertyName = propertyName;
            return this;
        }

        public Builder setOldValue(String oldValue) {
            this.oldValue = oldValue;
            return this;
        }

        public Builder setNewValue(String newValue) {
            this.newValue = newValue;
            return this;
        }

        public ImmutableAuditLogEntry build() {
            return new ImmutableAuditLogEntry(this.propertyName, this.oldValue, this.newValue);
        }
    }
}
