package com.atlassian.crowd.embedded.atlassianuser;

import com.atlassian.crowd.embedded.api.SearchRestriction;
import com.atlassian.crowd.search.builder.Restriction;
import com.atlassian.crowd.search.query.entity.restriction.BooleanRestriction;
import com.atlassian.crowd.search.query.entity.restriction.BooleanRestrictionImpl;
import com.atlassian.crowd.search.query.entity.restriction.MatchMode;
import com.atlassian.crowd.search.query.entity.restriction.Property;
import com.atlassian.crowd.search.query.entity.restriction.TermRestriction;
import com.atlassian.crowd.search.query.entity.restriction.constants.GroupTermKeys;
import com.atlassian.crowd.search.query.entity.restriction.constants.UserTermKeys;
import com.atlassian.user.search.query.BooleanQuery;
import com.atlassian.user.search.query.EmailTermQuery;
import com.atlassian.user.search.query.FullNameTermQuery;
import com.atlassian.user.search.query.GroupNameTermQuery;
import com.atlassian.user.search.query.Query;
import com.atlassian.user.search.query.TermQuery;
import com.atlassian.user.search.query.UserNameTermQuery;

import java.util.ArrayList;
import java.util.List;

/**
 * Converts Atlassian-User {@link Query} objects to Crowd search API {@link Restriction} objects.
 *
 * @see EmbeddedCrowdEntityQueryParser
 */
final class QueryRestrictionConverter
{
    SearchRestriction toRestriction(Query query)
    {
        if (query instanceof BooleanQuery)
        {
            return toBooleanRestriction((BooleanQuery) query);
        }

        assert query instanceof TermQuery : "There are only two basic types of queries: TermQuery and BooleanQuery";
        return toTermRestriction((TermQuery) query);
    }

    SearchRestriction toBooleanRestriction(BooleanQuery booleanQuery)
    {
        List<Query> nestedQueries = booleanQuery.getQueries();
        List<SearchRestriction> restrictions = new ArrayList<SearchRestriction>(nestedQueries.size());
        for (Query nestedQuery : nestedQueries)
        {
            restrictions.add(toRestriction(nestedQuery));
        }
        BooleanRestriction.BooleanLogic logic = booleanQuery.isAND() ?
                BooleanRestriction.BooleanLogic.AND :
                BooleanRestriction.BooleanLogic.OR;
        return new BooleanRestrictionImpl(logic, restrictions);
    }

    SearchRestriction toTermRestriction(TermQuery termQuery)
    {
        return new TermRestriction<String>(getTermQueryProperty(termQuery),
                getTermQueryMatchMode(termQuery), termQuery.getTerm());
    }

    private MatchMode getTermQueryMatchMode(TermQuery query)
    {
        String matchingRule = query.getMatchingRule();
        if (TermQuery.SUBSTRING_STARTS_WITH.equals(matchingRule))
        {
            return MatchMode.STARTS_WITH;
        }
        if (TermQuery.SUBSTRING_ENDS_WITH.equals(matchingRule))
        {
            return MatchMode.CONTAINS; // TODO - adds ENDS_WITH to Crowd search API
        }
        if (TermQuery.SUBSTRING_CONTAINS.equals(matchingRule))
        {
            return MatchMode.CONTAINS;
        }
        if (TermQuery.WILDCARD.equals(matchingRule))
        {
            throw new IllegalArgumentException("Wildcard queries are not accepted by Crowd");
        }
        return MatchMode.EXACTLY_MATCHES;
    }

    private Property<String> getTermQueryProperty(TermQuery query)
    {
        if (query instanceof UserNameTermQuery)
        {
            return UserTermKeys.USERNAME;
        }
        if (query instanceof EmailTermQuery)
        {
            return UserTermKeys.EMAIL;
        }
        if (query instanceof FullNameTermQuery)
        {
            return UserTermKeys.DISPLAY_NAME;
        }
        if (query instanceof GroupNameTermQuery)
        {
            return GroupTermKeys.NAME;
        }
        throw new IllegalArgumentException("Unknown query type: " + query);
    }
}
