package com.atlassian.crowd.embedded.atlassianuser;

import java.util.Set;

import com.atlassian.crowd.embedded.api.CrowdDirectoryService;
import com.atlassian.crowd.embedded.api.CrowdService;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.OperationType;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.exception.InvalidUserException;
import com.atlassian.crowd.exception.OperationNotPermittedException;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.util.SecureRandomStringUtils;
import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.User;
import com.atlassian.user.UserManager;
import com.atlassian.user.impl.DefaultUser;
import com.atlassian.user.impl.DuplicateEntityException;
import com.atlassian.user.impl.EntityMissingException;
import com.atlassian.user.impl.EntityValidationException;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.page.Pagers;
import com.atlassian.user.security.authentication.InvalidPasswordException;
import com.atlassian.user.security.password.Credential;
import com.atlassian.user.util.Assert;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import static com.atlassian.crowd.search.EntityDescriptor.user;
import static com.atlassian.crowd.search.query.entity.EntityQuery.ALL_RESULTS;
import static com.google.common.base.Preconditions.checkNotNull;

public final class EmbeddedCrowdUserManager implements UserManager
{
    private final static String DEFAULT_BLANK = "-";

    /**
     * When a user is created with a blank password, they are assigned a randomly generated
     * alphanumeric password of this length. Since alphanumeric passwords have 5.954 bits of
     * entropy per character, a 22-character password gives us the equivalent of a 130 bit
     * random number.
     *
     * The methods to create password-less users are all deprecated, and (in Confluence at least)
     * only called from test code.
     */
    private static final int RANDOM_PASSWORD_LENGTH = 22;

    private final RepositoryIdentifier repositoryIdentifier;
    private final CrowdService crowdService;
    private final CrowdDirectoryService crowdDirectoryService;

    public EmbeddedCrowdUserManager(final RepositoryIdentifier repositoryIdentifier, final CrowdService crowdService, final CrowdDirectoryService crowdDirectoryService)
    {
        this.repositoryIdentifier = checkNotNull(repositoryIdentifier);
        this.crowdService = checkNotNull(crowdService);
        this.crowdDirectoryService = checkNotNull(crowdDirectoryService);
    }

    @Override
    public RepositoryIdentifier getIdentifier()
    {
        return repositoryIdentifier;
    }

    @Override
    public RepositoryIdentifier getRepository(final Entity entity)
    {
        if (getUser(entity.getName()) != null)
        {
            return repositoryIdentifier;
        }
        else
        {
            return null; // entity not handled by this repository
        }
    }

    @Override
    public boolean isCreative()
    {
        return true;
    }

    @Override
    public Pager<User> getUsers()
    {
        final Iterable<com.atlassian.crowd.embedded.api.User> allUsers = crowdService.search(QueryBuilder.queryFor(com.atlassian.crowd.embedded.api.User.class, user()).returningAtMost(ALL_RESULTS));
        return Pagers.newDefaultPager(Iterables.transform(allUsers, Conversions.TO_ATLASSIAN_USER));
    }

    @Override
    public Pager<String> getUserNames()
    {
        Iterable<String> usernames = crowdService.search(QueryBuilder.queryFor(String.class, user()).returningAtMost(ALL_RESULTS));
        return new DefaultPager<String>(Lists.newArrayList(usernames.iterator()));
    }

    @Override
    public User getUser(String username)
    {
        Assert.notNull(username, "username must not be null.");
        return Conversions.TO_ATLASSIAN_USER.apply(crowdService.getUser(username));
    }

    @Override
    public User createUser(final String username) throws EntityException
    {
        DefaultUser userTemplate = new DefaultUser(username, DEFAULT_BLANK, DEFAULT_BLANK);
        return createUser(userTemplate, Credential.NONE);
    }

    @Override
    public User createUser(User userTemplate, Credential credential) throws EntityException, IllegalArgumentException
    {
        User existingUser = getUser(userTemplate.getName());
        if (existingUser != null)
        {
            throw new DuplicateEntityException("User with name [" + userTemplate.getName() + "] " +
                    "already exists in this repository (" + getIdentifier().getName() + ")");
        }

        // Crowd can not handle users with blank passwords, so we create a random password
        // instead.
        if (Credential.NONE.equals(credential))
        {
            credential = createRandomCredential();
        }

        if (credential.isEncrypted())
        {
            throw new IllegalArgumentException("Cannot create a user with an already encrypted credential");
        }

        com.atlassian.crowd.embedded.api.User crowdUser;
        try
        {
            crowdUser = crowdService.addUser(toUserTemplate(userTemplate), credential.getValue());
        }
        catch (InvalidUserException e)
        {
            throw new EntityValidationException(e);
        }
        catch (InvalidCredentialException e)
        {
            throw new InvalidPasswordException(e);
        }
        catch (OperationNotPermittedException e)
        {
            throw new EntityException(e);
        }

        return Conversions.TO_ATLASSIAN_USER.apply(crowdUser);
    }

    private Credential createRandomCredential()
    {
        String randomPassword = SecureRandomStringUtils.getInstance().randomAlphanumericString(RANDOM_PASSWORD_LENGTH);
        return Credential.unencrypted(randomPassword);
    }

    private UserTemplate toUserTemplate(User atlassianUser)
    {
        UserTemplate template = new UserTemplate(atlassianUser.getName());
        template.setDisplayName(atlassianUser.getFullName());
        template.setEmailAddress(atlassianUser.getEmail());
        template.setActive(true);
        return template;
    }

    @Override
    public void alterPassword(final User user, final String password) throws EntityException
    {
        try
        {
            crowdService.updateUserCredential(getCrowdUser(user), password);
        }
        catch (com.atlassian.crowd.exception.runtime.UserNotFoundException e)
        {
            throw new EntityMissingException(e);
        }
        catch (InvalidCredentialException e)
        {
            throw new InvalidPasswordException(e);
        }
        catch (OperationNotPermittedException e)
        {
            throw new EntityException(e);
        }
    }

    @Override
    public void saveUser(final User user) throws EntityException, IllegalArgumentException
    {
        try
        {
            crowdService.updateUser(toUserTemplate(user));
        }
        catch (InvalidUserException e)
        {
            throw new EntityException(e);
        }
        catch (OperationNotPermittedException e)
        {
            throw new EntityException(e);
        }
    }

    @Override
    public void removeUser(final User user) throws EntityException, IllegalArgumentException
    {
        try
        {
            final com.atlassian.crowd.embedded.api.User crowdUser = getCrowdUser(user);
            if (crowdUser != null)
            {
                crowdService.removeUser(crowdUser);
            }
            else
            {
                throw new IllegalArgumentException("User [" + user.getName() + "] is not managed by embedded crowd");
            }
        }
        catch (OperationNotPermittedException e)
        {
            throw new EntityException(e);
        }
    }

    private com.atlassian.crowd.embedded.api.User getCrowdUser(User user) throws IllegalArgumentException
    {
        Assert.notNull(user, "User should not be null");
        if (user instanceof com.atlassian.crowd.model.user.User)
        {
            return (com.atlassian.crowd.model.user.User) user;
        }
        return crowdService.getUser(user.getName());
    }

    @Override
    public boolean isReadOnly(final User user)
    {
        com.atlassian.crowd.embedded.api.User crowdUser = getCrowdUser(user);
        if (crowdUser == null)
        {
            return false;
        }
        else
        {
            Directory directory = crowdDirectoryService.findDirectoryById(crowdUser.getDirectoryId());
            Set<OperationType> allowedOperations = directory.getAllowedOperations();
            // read-only directories don't permit creating users
            return !allowedOperations.contains(OperationType.CREATE_USER);
        }
    }
}
