package com.atlassian.crowd.embedded.atlassianuser;

import com.atlassian.crowd.embedded.api.User;
import org.apache.commons.lang.builder.HashCodeBuilder;

import static com.atlassian.crowd.embedded.impl.IdentifierUtils.toLowerCase;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * An implementation of both the Atlassian-User {@link com.atlassian.user.User} interface and the Crowd {@link User} interface, used to bridge the two APIs.
 * <p>
 * Where the contracts of the two APIs differ, like with {@link #equals(Object)} behaviour, the Crowd API is followed.
 * </p>
 */
public class EmbeddedCrowdUser implements com.atlassian.user.User, com.atlassian.crowd.embedded.api.User
{
    // not using delegation because setters need to modify some state
    private final boolean active;

    private final String name;
    private final long directoryId;
    private final String emailAddress;
    private final String displayName;

    /**
     * Constructs a new EmbeddedCrowdUser with the specified properties. The name must not be null.
     *
     * @throws NullPointerException if the provided name is null
     */
    public EmbeddedCrowdUser(long directoryId, String name, String displayName, String emailAddress, boolean isActive)
    {
        this.directoryId = directoryId;
        this.name = checkNotNull(name, "Username must not be null");
        this.displayName = displayName;
        this.emailAddress = emailAddress;
        this.active = isActive;
    }

    /**
     * Constructs a new active EmbeddedCrowdUser with the specified properties. The name must not be null.
     * The user's active property will be set to true.
     *
     * @throws NullPointerException if the provided name is null
     */
    public EmbeddedCrowdUser(long directoryId, String name, String displayName, String emailAddress)
    {
        this(directoryId, name, displayName, emailAddress, true);
    }

    /**
     * Constructs a new Embedded Crowd user with a <em>copy</em> of the properties from the provided user.
     *
     * @param crowdUser a user from Crowd, which must not be {@code null} and must not have a {@code null} name
     * @throws NullPointerException if the user or its name is {@code null}
     */
    public EmbeddedCrowdUser(User crowdUser)
    {
        this(checkNotNull(crowdUser, "User must not be null").getDirectoryId(),
            crowdUser.getName(),
            crowdUser.getDisplayName(),
            crowdUser.getEmailAddress(),
            crowdUser.isActive());
    }

    @Override
    public String getFullName()
    {
        return getDisplayName();
    }

    @Override
    public String getEmail()
    {
        return getEmailAddress();
    }

    @Override
    public boolean isActive()
    {
        return active;
    }

    @Override
    public String getEmailAddress()
    {
        return emailAddress;
    }

    @Override
    public String getDisplayName()
    {
        return displayName;
    }

    @Override
    public int compareTo(com.atlassian.crowd.embedded.api.User user)
    {
        return getName().compareToIgnoreCase(user.getName());
    }

    @Override
    public final long getDirectoryId()
    {
        return directoryId;
    }

    @Override
    public final String getName()
    {
        return name;
    }

    private String getLowerName()
    {
        return toLowerCase(name);
    }

    @Override
    public String toString()
    {
        return String.format("EmbeddedCrowdUser{name='%s', displayName='%s', directoryId=%d}",
            name, displayName, directoryId);
    }

    @Override
    public final boolean equals(final Object o)
    {
        if (this == o)
        {
            return true;
        }
        if (o == null || getClass() != o.getClass())
        {
            return false;
        }

        EmbeddedCrowdUser that = (EmbeddedCrowdUser) o;
        return directoryId == that.directoryId && getLowerName().equals(that.getLowerName());
    }

    @Override
    public final int hashCode()
    {
        return new HashCodeBuilder(1, 31).append(directoryId).append(getLowerName()).toHashCode();
    }
}
