package com.atlassian.crowd.embedded.atlassianuser;

import com.atlassian.crowd.embedded.api.CrowdService;
import com.atlassian.crowd.exception.OperationNotPermittedException;
import com.atlassian.crowd.exception.embedded.InvalidGroupException;
import com.atlassian.crowd.exception.runtime.CrowdRuntimeException;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.user.*;
import com.atlassian.user.Group;
import com.atlassian.user.User;
import com.atlassian.user.impl.EntityValidationException;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.page.Pagers;
import com.atlassian.user.util.Assert;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import java.util.List;

import static com.atlassian.crowd.search.EntityDescriptor.group;
import static com.atlassian.crowd.search.EntityDescriptor.user;
import static com.atlassian.crowd.search.query.entity.EntityQuery.ALL_RESULTS;
import static com.google.common.base.Preconditions.checkNotNull;

public class EmbeddedCrowdGroupManager implements GroupManager
{
    private final RepositoryIdentifier repositoryIdentifier;
    private final CrowdService crowdService;

    public EmbeddedCrowdGroupManager(final RepositoryIdentifier repositoryIdentifier, final CrowdService crowdService)
    {
        this.repositoryIdentifier = checkNotNull(repositoryIdentifier);
        this.crowdService = checkNotNull(crowdService);
    }

    @Override
    public RepositoryIdentifier getIdentifier()
    {
        return repositoryIdentifier;
    }

    @Override
    public RepositoryIdentifier getRepository(final Entity entity) throws EntityException
    {
        if (getGroup(entity.getName()) != null)
        {
            return repositoryIdentifier;
        }
        else
        {
            // entity not handled by this repository
            return null;
        }
    }

    @Override
    public boolean isCreative()
    {
        return true;
    }

    @Override
    public Pager<Group> getGroups() throws EntityException
    {
        Iterable<com.atlassian.crowd.embedded.api.Group> allGroups = crowdService.search(Queries.ALL_GROUPS);
        return Pagers.newDefaultPager(Iterables.transform(allGroups, Conversions.TO_ATLASSIAN_GROUP));
    }

    @Override
    public List<Group> getWritableGroups()
    {
        Iterable<com.atlassian.crowd.embedded.api.Group> allGroups = crowdService.search(Queries.ALL_GROUPS);
        return Lists.newArrayList(Iterables.transform(allGroups, Conversions.TO_ATLASSIAN_GROUP));
    }

    @Override
    public Pager<Group> getGroups(final User user) throws EntityException
    {
        Iterable<com.atlassian.crowd.embedded.api.Group> groupsForUser = crowdService.search(Queries.groupsForUser(user));
        return Pagers.newDefaultPager(Iterables.transform(groupsForUser, Conversions.TO_ATLASSIAN_GROUP));
    }

    @Override
    public Pager<String> getMemberNames(final Group group) throws EntityException
    {
        Iterable<String> members = crowdService.search(Queries.groupMemberNames(group));
        return Pagers.newDefaultPager(members);
    }

    @Override
    public Pager<String> getLocalMemberNames(final Group group) throws EntityException
    {
        return getMemberNames(group);
    }

    @Override
    public Pager<String> getExternalMemberNames(final Group group) throws EntityException
    {
        return DefaultPager.emptyPager();
    }

    @Override
    public Group getGroup(final String groupName) throws EntityException
    {
        if (groupName == null)
        {
            // same logic as HibernateGroupManager from atlassian-user
            throw new IllegalArgumentException("Input (groupname) is null.");
        }

        return Conversions.TO_ATLASSIAN_GROUP.apply(crowdService.getGroup(groupName));
    }

    @Override
    public Group createGroup(final String groupName) throws EntityException
    {
        // TODO Replace this with a 'real' template class
        com.atlassian.crowd.embedded.api.Group template = new com.atlassian.crowd.embedded.api.Group()
        {
            @Override
            public String getName()
            {
                return groupName;
            }

            @Override
            public int compareTo(com.atlassian.crowd.embedded.api.Group group)
            {
                return getName().compareToIgnoreCase(group.getName());
            }
        };

        com.atlassian.crowd.embedded.api.Group group;
        try
        {
            group = crowdService.addGroup(template);
        }
        catch (InvalidGroupException e)
        {
            throw new EntityValidationException(e);
        }
        catch (OperationNotPermittedException e)
        {
            throw new EntityException(e);
        }

        return Conversions.TO_ATLASSIAN_GROUP.apply(group);
    }

    @Override
    public void removeGroup(final Group group) throws EntityException, IllegalArgumentException
    {
        try
        {
            crowdService.removeGroup(getCrowdGroup(group));
        }
        catch (OperationNotPermittedException e)
        {
            throw new EntityException(e);
        }
    }

    @Override
    public void addMembership(final Group group, final User user) throws EntityException, IllegalArgumentException
    {
        com.atlassian.crowd.embedded.api.Group crowdGroup = getCrowdGroup(group);
        if (crowdGroup == null)
            throw new EntityException("This repository [" + getIdentifier().getName() + "] does not handle memberships of the group [" + group + "]");

        com.atlassian.crowd.embedded.api.User crowdUser = getCrowdUser(user);
        if (crowdUser == null)
            throw new EntityException("This repository [" + getIdentifier().getName() + "] does not handle memberships of the user [" + user + "]");

        try
        {
            crowdService.addUserToGroup(crowdUser, crowdGroup);
        }
        catch (OperationNotPermittedException e)
        {
            throw new EntityException(e);
        }
    }

    @Override
    public boolean hasMembership(final Group group, final User user) throws EntityException
    {
        com.atlassian.crowd.embedded.api.Group crowdGroup = getCrowdGroup(group);
        if (crowdGroup == null)
            return false;

        com.atlassian.crowd.embedded.api.User crowdUser = getCrowdUser(user);
        if (crowdUser == null)
            return false;

        return crowdService.isUserMemberOfGroup(crowdUser, crowdGroup);
    }

    @Override
    public void removeMembership(final Group group, final User user) throws EntityException, IllegalArgumentException
    {
        com.atlassian.crowd.embedded.api.Group crowdGroup = getCrowdGroup(group);
        if (crowdGroup == null)
            throw new EntityException("This repository [" + getIdentifier().getName() + "] does not handle memberships of the group [" + group + "]");

        com.atlassian.crowd.embedded.api.User crowdUser = getCrowdUser(user);
        if (crowdUser == null)
            throw new EntityException("This repository [" + getIdentifier().getName() + "] does not handle memberships of the user [" + user + "]");

        try
        {
            crowdService.removeUserFromGroup(crowdUser, crowdGroup);
        }
        catch (OperationNotPermittedException e)
        {
            throw new EntityException(e);
        }
        catch (CrowdRuntimeException e)
        {
            throw new EntityException(e);
        }
    }

    private com.atlassian.crowd.embedded.api.User getCrowdUser(User user)
    {
        Assert.notNull(user, "User should not be null");
        if (user instanceof com.atlassian.crowd.model.user.User)
            return (com.atlassian.crowd.model.user.User) user;
        return crowdService.getUser(user.getName());
    }

    private com.atlassian.crowd.embedded.api.Group getCrowdGroup(Group group)
    {
        Assert.notNull(group, "Group should not be null");
        if (group instanceof com.atlassian.crowd.embedded.api.Group)
            return (com.atlassian.crowd.embedded.api.Group) group;
        return crowdService.getGroup(group.getName());
    }

    @Override
    public boolean supportsExternalMembership() throws EntityException
    {
        return false;
    }

    @Override
    public boolean isReadOnly(final Group group) throws EntityException
    {
        return false;
    }

    private static abstract class Queries
    {
        private static final EntityQuery<com.atlassian.crowd.embedded.api.Group> ALL_GROUPS = QueryBuilder
            .queryFor(com.atlassian.crowd.embedded.api.Group.class, group())
            .returningAtMost(ALL_RESULTS);

        private static MembershipQuery<com.atlassian.crowd.embedded.api.Group> groupsForUser(User user)
        {
            return QueryBuilder.queryFor(com.atlassian.crowd.embedded.api.Group.class, group())
                .parentsOf(user())
                .withName(user.getName())
                .returningAtMost(ALL_RESULTS);
        }

        private static MembershipQuery<String> groupMemberNames(Group group)
        {
            return QueryBuilder.queryFor(String.class, user())
                .childrenOf(group())
                .withName(group.getName())
                .returningAtMost(ALL_RESULTS);
        }
    }
}
