package com.atlassian.crowd.embedded.atlassianuser;

import com.atlassian.crowd.embedded.api.*;
import com.atlassian.crowd.embedded.api.Group;
import com.atlassian.crowd.embedded.api.User;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.user.*;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.SearchResult;
import com.atlassian.user.search.query.*;
import com.atlassian.user.search.query.Query;

import static com.atlassian.crowd.search.EntityDescriptor.group;
import static com.atlassian.crowd.search.EntityDescriptor.user;
import static com.atlassian.crowd.search.query.entity.EntityQuery.ALL_RESULTS;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Implementation of Atlassian-User search API which converts queries into the Crowd search API
 * and runs them against the {@link com.atlassian.crowd.embedded.api.CrowdService}.
 *
 * @see EntityQueryParser
 * @see QueryRestrictionConverter
 * @see CrowdSearchResult
 */
public final class EmbeddedCrowdEntityQueryParser implements EntityQueryParser
{
    private final RepositoryIdentifier repositoryIdentifier;
    private final CrowdService crowdService;

    private final QueryValidator queryValidator = new QueryValidator();
    private final QueryRestrictionConverter queryRestrictionConverter = new QueryRestrictionConverter();

    public EmbeddedCrowdEntityQueryParser(final RepositoryIdentifier repositoryIdentifier, CrowdService crowdService)
    {
        this.repositoryIdentifier = checkNotNull(repositoryIdentifier);
        this.crowdService = checkNotNull(crowdService);
    }

    @Override
    public SearchResult<com.atlassian.user.User> findUsers(Query query) throws EntityQueryException
    {
        queryValidator.assertValid(query);
        SearchRestriction restriction = queryRestrictionConverter.toRestriction(query);
        com.atlassian.crowd.embedded.api.Query<User> crowdQuery = QueryBuilder.queryFor(
            User.class, user(), restriction, 0, ALL_RESULTS);
        return CrowdSearchResult.forUsers(repositoryIdentifier, crowdService.search(crowdQuery));
    }

    @Override
    public SearchResult<com.atlassian.user.Group> findGroups(Query query) throws EntityException
    {
        queryValidator.assertValid(query);
        SearchRestriction restriction = queryRestrictionConverter.toRestriction(query);
        com.atlassian.crowd.embedded.api.Query<Group> crowdQuery = QueryBuilder.queryFor(
            Group.class, group(), restriction, 0, ALL_RESULTS);
        return CrowdSearchResult.forGroups(repositoryIdentifier, crowdService.search(crowdQuery));
    }

    @Override
    public SearchResult<com.atlassian.user.User> findUsers(Query query, QueryContext context) throws EntityException
    {
        if (!context.contains(repositoryIdentifier))
            return null;

        return findUsers(query);
    }

    @Override
    public SearchResult<com.atlassian.user.Group> findGroups(Query query, QueryContext context) throws EntityException
    {
        if (!context.contains(repositoryIdentifier))
            return null;

        return findGroups(query);
    }
}
