/**
 * Stickytable headers for confluence
 */
define("sticky-table-headers/viewpage", [
    "jquery",
    "ajs",
    "sticky-table-headers/utils"
], function(
    $,
    AJS,
    Utils
) {
    "use strict";

    var previousObserver;
    var previousOptionsHandler;

    /**
     * setupTable - Enable sticky tables for a group of table elements
     *
     * @param {HTMLElement|JqueryHtmlElement} table Setup stickyheaders for tables with a defined parameter for valid tables
     * @param {Object} options StickyTableHeaders jquery plugin options
     */
    var setupTable = function(table, options) {
        // setup sticky table headers
        var $validTables = $(table).filter(function() {
            return Utils.isAValidTable(this);
        });

        $validTables.stickyTableHeaders(options).addClass('stickyTableHeaders');

        var optionsHandler = function (event, newOptions) {
          $validTables.stickyTableHeaders(newOptions);
        };
        if (previousOptionsHandler) {
          AJS.unbind('sticky-table-headers.change.options', previousOptionsHandler);
        }
        AJS.bind('sticky-table-headers.change.options', optionsHandler);
        previousOptionsHandler = optionsHandler;

        Utils.bindHorizontalScrolling($validTables);
    };

    /**
     * initialize stickyheaders for all confluence tables
     *
     */
    var initialize = function() {
        var $table = $(".confluenceTable:not(.stickyTableHeaders)");
        var $win = $(window);

        // We need to abort ASAP if we're in the documentation space
        if (!Utils.isNotDocumentationTheme()) {
            return;
        }

        // the editor structure by default does not have a thead
        Utils.moveHeaderRowsToTHead($table);

        // We use mutation observer in order to trigger resizing
        if (window.MutationObserver) {

            // we need to listen to all outer container that could cause reflow
            var target = document.querySelector("#page") || document.querySelector("#confluence-ui");

            // we need to exclude stickyheaders own mutations
            var stickyHeadersMutationTarget = /sticky|tableFloating|confluenceTh/;

            // listen for changes in order to make sure tableHeaders are repositioned upon reflow
            var observer = new MutationObserver(
                $.debounce(function(mutations) {
                    var validtriggerFlag = false;
                    var className;

                    for (var i = 0, len = mutations.length; i < len; i++) {
                        className = mutations[i].target.className;

                        if (!stickyHeadersMutationTarget.test(className)) {
                            validtriggerFlag = true;
                            break;
                        }
                    }

                    if (validtriggerFlag) {
                        $win.trigger('resize.stickyTableHeaders');
                    }
                }, 0)
            );

            // start observing
            if (previousObserver && previousObserver.unobserve) {
              previousObserver.unobserve();
            }
            observer.observe(target, {
                subtree: true,
                attributes: true,
                childList: true
            });
            previousObserver = observer;
        }

        // enable all confluence tables
        setupTable($table, {
            cacheHeaderHeight: true
        });
    };

    return {
        initialize: initialize,
        setupTable: setupTable
    };
});
