package com.atlassian.confluence.plugins.jirareports;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.ApplicationLinkRequest;
import com.atlassian.applinks.api.ApplicationLinkRequestFactory;
import com.atlassian.applinks.api.ApplicationLinkService;
import com.atlassian.applinks.api.CredentialsRequiredException;
import com.atlassian.applinks.api.ReadOnlyApplicationLink;
import com.atlassian.applinks.api.auth.Anonymous;
import com.atlassian.plugins.rest.common.security.AnonymousAllowed;
import com.atlassian.sal.api.net.Request;
import com.atlassian.sal.api.net.ResponseException;
import org.apache.commons.httpclient.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

/**
 * The service retrieves jira servers, projects and versions
 * REST URL: /jirareports/1.0/
 *
 * @since 2.1
 */
@Path("/")
@Produces({MediaType.APPLICATION_JSON})
@AnonymousAllowed
public class JiraReportsResource {

    private static final String PROJECT_REST_URI = "/rest/api/2/project";
    private static final Logger LOG = LoggerFactory.getLogger(JiraReportsResource.class);

    private final ApplicationLinkService appLinkService;

    public JiraReportsResource(ApplicationLinkService appLinkService) {
        this.appLinkService = appLinkService;
    }

    /**
     * The service retrieves all projects of server.
     * REST URL: /jirareports/1.0/appLink/{appLinkId}/projects
     *
     * @param appLinkId applink id to gain jira server
     * @return list of projects
     */
    @GET
    @Path("appLink/{appLinkId}/projects")
    public Response getProjectsByAppLinkId(@PathParam("appLinkId") String appLinkId) {
        ReadOnlyApplicationLink appLink = null;
        String url = null;
        try {
            appLink = appLinkService.getApplicationLink(new ApplicationId(appLinkId));
            url = appLink.getRpcUrl() + PROJECT_REST_URI;
            String projects = requestJiraByAuthenticatedUser(appLink, url);
            return Response.ok(projects).build();
        } catch (CredentialsRequiredException e) {
            String projects = requestJiraByAnonymousUser(appLink, url);
            return buildUnAuthenticatedResponse(projects, e.getAuthorisationURI().toString());
        } catch (Exception e) {
            LOG.error("Can not retrieve projects", e);
            return Response.status(HttpStatus.SC_BAD_REQUEST).build();
        }
    }

    /**
     * The service retrieves all versions of project.
     * REST URL: /jirareports/1.0/appLink/{appLinkId}/appLink/{appLinkId}/project/{projectKey}/versions
     *
     * @param appLinkId  applink id to gain jira server
     * @param projectKey project key is maps in jira server
     * @return list of versions
     */
    @GET
    @Path("appLink/{appLinkId}/project/{projectKey}/versions")
    public Response getVersionsByKeyProject(@PathParam("appLinkId") String appLinkId,
                                            @PathParam("projectKey") String projectKey) {
        ReadOnlyApplicationLink appLink = null;
        String url = null;
        try {
            appLink = appLinkService.getApplicationLink(new ApplicationId(appLinkId));
            url = appLink.getRpcUrl() + PROJECT_REST_URI + "/" + projectKey + "/versions";
            String versions = requestJiraByAuthenticatedUser(appLink, url);
            return Response.ok(versions).build();
        } catch (CredentialsRequiredException e) {
            String versions = requestJiraByAnonymousUser(appLink, url);
            return buildUnAuthenticatedResponse(versions, e.getAuthorisationURI().toString());
        } catch (Exception e) {
            LOG.error("Can not retrieve versions", e);
            return Response.status(HttpStatus.SC_BAD_REQUEST).build();
        }
    }

    /**
     * This function build unAuthentication response
     *
     * @param responseData     data retrieve by anonymous user
     * @param authorisationURI link to map user
     * @return response with status unAuthentication
     */
    private Response buildUnAuthenticatedResponse(String responseData, String authorisationURI) {
        Response.ResponseBuilder response = Response.status(HttpStatus.SC_UNAUTHORIZED)
                .header("WWW-Authenticate", "OAuth realm=\"" + authorisationURI + "\"");

        if (responseData != null) {
            response.entity(responseData);
        }

        return response.build();
    }

    /**
     * Request jira by anonymous user
     *
     * @param appLink jira server app link
     * @param url     rest url
     * @return response
     */
    private String requestJiraByAnonymousUser(ReadOnlyApplicationLink appLink, String url) {
        try {
            final ApplicationLinkRequestFactory requestFactory = appLink.createAuthenticatedRequestFactory(Anonymous.class);
            ApplicationLinkRequest request = requestFactory.createRequest(Request.MethodType.GET, url);
            return request.execute();
        } catch (Exception e) {
            LOG.error("Can not retrieve data from jira server by anonymous user", e);
            return null;
        }
    }

    /**
     * Request jira by login user
     *
     * @param appLink jira server app link
     * @param url     rest url
     * @return response
     * @throws CredentialsRequiredException when user is not mapping
     * @throws ResponseException            when have problem request jira server
     */
    private String requestJiraByAuthenticatedUser(ReadOnlyApplicationLink appLink, String url)
            throws CredentialsRequiredException, ResponseException {
        final ApplicationLinkRequestFactory requestFactory = appLink.createAuthenticatedRequestFactory();
        ApplicationLinkRequest request = requestFactory.createRequest(Request.MethodType.GET, url);
        return request.execute();
    }
}
