package com.atlassian.confluence.plugins.jirareports;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.ApplicationLinkService;
import com.atlassian.applinks.api.ReadOnlyApplicationLink;
import com.atlassian.confluence.extra.jira.JiraIssuesManager;
import com.atlassian.confluence.plugins.SoftwareBlueprintsContextProviderHelper;
import com.atlassian.confluence.plugins.common.event.SoftwareBPAnalyticEventUtils;
import com.atlassian.confluence.plugins.createcontent.api.contextproviders.AbstractBlueprintContextProvider;
import com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContext;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URLEncoder;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;

public class StatusReportContextProvider extends AbstractBlueprintContextProvider {
    private static final Logger LOGGER = LoggerFactory.getLogger(StatusReportContextProvider.class);
    private static final String SOY_PIE_CHART_MACRO_TEMPLATE = "Confluence.Blueprints.JiraReports.Template.piechart.soy";
    private static final String SOY_JIRAISSUES_MACRO_TEMPLATE = "Confluence.Blueprints.JiraReports.Template.jiraissues.soy";

    private SoftwareBlueprintsContextProviderHelper helper;
    private ApplicationLinkService appLinkService;
    private JiraIssuesHelper jiraIssuesHelper;

    @Override
    protected BlueprintContext updateBlueprintContext(BlueprintContext context) {
        String appId = (String) context.get("jira-reports-servers");

        if (StringUtils.isNotBlank(appId)) {
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat(helper.getDateFormat(), helper.getAuthenticatedUserLocale());
            context.put("date", simpleDateFormat.format(new Date()));

            try {
                ReadOnlyApplicationLink appLink = appLinkService.getApplicationLink(new ApplicationId(appId));
                if (appLink == null) {
                    return context;
                }

                String jql = null;
                if (("jirareports-statusreport-static").equals(context.get("statusreport-type"))) {
                    jql = URLEncoder.encode(jiraIssuesHelper.buildProjectVersionJQL(context.getMap()), "UTF-8");
                } else {
                    jql = (String) context.get("jira-query");
                }

                //for dynamic report
                if ("dynamic".equals(context.get("issues-list-type")) && "jirareports-statusreport-dynamic".equals(context.get("statusreport-type"))) {
                    String totalIssues = getJiraIssuesMacro("jqlQuery", context.get("jira-query"), appLink, true);
                    context.put("totalIssues", totalIssues);

                    context.put("pie_chart_summary", renderChart(jql, "statuses", appLink, helper.getText("jirareports.statusreport.blueprint.form.statType.status"), totalIssues));
                    context.put("pie_chart_priority", renderChart(jql, "priorities", appLink, helper.getText("jirareports.statusreport.blueprint.form.statType.priority"), totalIssues));
                    context.put("pie_chart_component", renderChart(jql, "components", appLink, helper.getText("jirareports.statusreport.blueprint.form.statType.component"), totalIssues));
                    context.put("pie_chart_issuetype", renderChart(jql, "issuetype", appLink, helper.getText("jirareports.statusreport.blueprint.form.statType.issuetype"), totalIssues));
                } else {
                    JiraIssuesManager.Channel channel = jiraIssuesHelper.getChannel(appLink, jql, 0);
                    int totalIssues = jiraIssuesHelper.getTotalIssueNumber(channel);
                    context.put("totalIssues", totalIssues + " issues");
                    //TODO: implement attachement image for snapshot status report
                }
            } catch (Exception e) {
                LOGGER.error("error render content", e);
            }
        } else {
            LOGGER.info("appId is null or empty");
        }
        doAnalytic(context);
        return context;
    }

    private String getJiraIssuesMacro(String keyJQL, Object valJQL, ReadOnlyApplicationLink appLink, boolean isCount) {
        HashMap<String, Object> jiraIssuesMacroContext = Maps.newHashMap();
        jiraIssuesMacroContext.put("serverId", appLink.getId().toString());
        jiraIssuesMacroContext.put("server", appLink.getName());
        jiraIssuesMacroContext.put("keyJQL", keyJQL);
        jiraIssuesMacroContext.put("valJQL", valJQL);
        jiraIssuesMacroContext.put("isCount", isCount);
        return helper.renderFromSoy(SoftwareBlueprintsContextProviderHelper.PLUGIN_KEY, SOY_JIRAISSUES_MACRO_TEMPLATE, jiraIssuesMacroContext);
    }

    private String renderChart(String jql, String statisticType, ReadOnlyApplicationLink appLink, String titleType, String totalIssues) {
        HashMap<String, Object> piechartContext = Maps.newHashMap();
        piechartContext.put("totalIssues", totalIssues);
        piechartContext.put("titleType", titleType);
        piechartContext.put("serverId", appLink.getId().toString());
        piechartContext.put("server", appLink.getName());
        piechartContext.put("jql", jql);
        piechartContext.put("statType", statisticType);
        piechartContext.put("width", "");
        piechartContext.put("border", true);

        return helper.renderFromSoy(SoftwareBlueprintsContextProviderHelper.PLUGIN_KEY, SOY_PIE_CHART_MACRO_TEMPLATE, piechartContext);
    }

    public void setHelper(SoftwareBlueprintsContextProviderHelper helper) {
        this.helper = helper;
    }

    public void setAppLinkService(ApplicationLinkService appLinkService) {
        this.appLinkService = appLinkService;
    }

    public void setJiraIssuesHelper(JiraIssuesHelper jiraIssuesHelper) {
        this.jiraIssuesHelper = jiraIssuesHelper;
    }

    private void doAnalytic(BlueprintContext context) {
        helper.publishAnalyticEvent(SoftwareBPAnalyticEventUtils.STATUS_REPORT_CREATE_EVENT_NAME);
        helper.publishAnalyticEvent(SoftwareBPAnalyticEventUtils.STATUS_REPORT_CREATE_DYNAMIC_EVENT_NAME);
        if ("simple".equals(context.get("dialogMode"))) {
            helper.publishAnalyticEvent(SoftwareBPAnalyticEventUtils.STATUS_REPORT_CREATE_SIMPLE_EVENT_NAME);
        }
    }

}
