define("cp/confluence/conversion-poller-backend",
    [
        "jquery",
        "underscore",
        "ajs"
    ],
    function(
        $,
        _,
        AJS
        ) {
        "use strict";

        function ConversionPollerBackend(type) {
            this._type = type;

            if (this._type === "thumbnail") {
                this._bulkUrl = AJS.contextPath() + "/rest/documentConversion/0.1/conversion/thumbnail/results";
                this._singleUrlBase = AJS.contextPath() + "/rest/documentConversion/0.1/conversion/thumbnail/";
            } else {
                this._bulkUrl = AJS.contextPath() + "/rest/documentConversion/0.1/conversion/convert/results";
                this._singleUrlBase = AJS.contextPath() + "/rest/documentConversion/0.1/conversion/convert/";
            }

            this._pollers = {};
            this._timerId = 0;
            this._interval = ConversionPollerBackend.INITIAL_INTERVAL;

            this._debouncedChange = _.debounce(this._change, 100); // allow for poller creation in bulk
        }

        ConversionPollerBackend.INITIAL_INTERVAL = 1000;
        ConversionPollerBackend.MAX_INTERVAL = 10000;
        ConversionPollerBackend.BACKOFF_PERCENT = 1.5;

        ConversionPollerBackend.prototype.add = function (attachmentId, version) {
            if (attachmentId && !this._pollers[attachmentId]) {
                var dfd = $.Deferred();
                this._pollers[attachmentId] = {_dfd: dfd, _version: version};
                this._debouncedChange();
            }
            return this._pollers[attachmentId]._dfd.promise();
        };

        ConversionPollerBackend.prototype.remove = function (attachmentId) {
            var poller = this._pollers[attachmentId];
            if (poller) {
                poller._dfd.reject("cancelled");
                delete this._pollers[attachmentId];
            }
        };


        ConversionPollerBackend.prototype._doPoll = function() {
            if (this._pollType === "single") {
                this._pollSingle();
            } else if (this._pollType === "multiple") {
                this._pollMultiple();
            }
        };


        ConversionPollerBackend.prototype._backoff = function () {
            var interval = this._interval * ConversionPollerBackend.BACKOFF_PERCENT;
            this._interval = interval > ConversionPollerBackend.MAX_INTERVAL ? ConversionPollerBackend.MAX_INTERVAL : interval;
            this._timerId = setTimeout(this._change.bind(this), this._interval);
        };

        ConversionPollerBackend.prototype._change = function () {
            var attachmentIds = this._getAttachmentIds();
            var hasAttachments = !(attachmentIds.length === 0);

            this._cancel();
            if (!hasAttachments) {
                delete this._pollType;
                this._interval = 1000;
            } else {
                this._changeType();
                this._doPoll();
            }
        };

        ConversionPollerBackend.prototype._changeType = function() {
            var attachmentIds = this._getAttachmentIds();
            var isSingle = attachmentIds.length === 1;
            var isMultiple = attachmentIds.length > 1;

            if (isSingle && this._pollType !== "single") {
                this._pollType = "single";
            } else if (isMultiple && this._pollType !== "multiple") {
                this._pollType = "multiple";
            }
        };

        ConversionPollerBackend.prototype._cancel = function () {
            clearTimeout(this._timerId);
            this._xhr && this._xhr.abort();
            delete this._xhr;
        };

        ConversionPollerBackend.prototype._getAttachmentIds = function () {
            return Object.keys(this._pollers);
        };

        ConversionPollerBackend.prototype._getIdsAndVersions = function () {
            var pollers = this._pollers;
            return _.map(this._getAttachmentIds(), function (attachmentId) {
                return {id:attachmentId, v:pollers[attachmentId]._version}
            });
        };

        ConversionPollerBackend.prototype._pollSingle = function () {
            var attachmentId = this._getAttachmentIds()[0];
            var poller = this._pollers[attachmentId];
            var version = poller._version;

            this._xhr = this._getSingle(attachmentId, version)
                .always(function () {
                    delete this._xhr;
                }.bind(this))
                .done(function(data, textStatus, jqXHR) {
                    if (jqXHR.status === 202) {
                        this._backoff();
                        poller._dfd.notify("converting");
                    } else {
                        poller._dfd.resolve(this._singleUrlBase + attachmentId + '/' + version, jqXHR.getResponseHeader("Content-Type"));
                        delete this._pollers[attachmentId];
                        this._change();
                    }
                }.bind(this))
                .fail(function(jqXHR, textStatus, errorThrown) {
                    if (jqXHR.status == 429) {
                        this._backoff();
                        poller._dfd.notify("busy");
                        return;
                    }

                    if (errorThrown === "abort") return;
                    poller._dfd.reject();
                    delete this._pollers[attachmentId];
                    this._change();
                }.bind(this));
        };

        ConversionPollerBackend.prototype._getSingle = function(attachmentId, version) {
            return $.ajax(this._singleUrlBase + attachmentId + '/' + version, {
                type: 'HEAD',
                dataType: "json"
            });
        };

        ConversionPollerBackend.prototype._pollMultiple = function () {
            this._xhr = $.ajax(this._bulkUrl, {
                type: 'POST',
                dataType: "json",
                contentType: "application/json; charset=utf-8",
                data: JSON.stringify(this._getIdsAndVersions())
            }).always(function () {
                delete this._xhr;
            }.bind(this))
            .done(function(statuses) {
                _(statuses).each(function(status, attachmentId) {
                    var poller = this._pollers[attachmentId];
                    if (!poller) { return; }

                    if (status == 200) {
                        var version = poller._version;
                        if (this._type !== "thumbnail") {
                            this._getSingle(attachmentId, version).done(function (data, textStatus, jqXHR) {
                                poller._dfd.resolve(this._singleUrlBase + attachmentId + '/' + version, jqXHR.getResponseHeader("Content-Type"));
                                delete this._pollers[attachmentId];
                            }.bind(this));
                        } else {
                            poller._dfd.resolve(this._singleUrlBase + attachmentId + '/' + version, "image/jpg");
                            delete this._pollers[attachmentId];
                        }
                    } else if (status >= 400 && status != 429) {
                        poller._dfd.reject();
                        delete this._pollers[attachmentId];
                    }
                }.bind(this));

                this._backoff();
            }.bind(this));
        };

        return ConversionPollerBackend;
    });