define(
    "cp/component/annotation/reply-view",
    [
        "backbone",
        "underscore",
        "jquery",
        "cp/component/annotation/reply",
        "cp/component/utils/editor-view",
        "cp/component/annotation/likes-view",
        "MediaViewer",
        "core/template-store-singleton"
    ],
    function(
        Backbone,
        _,
        $,
        Reply,
        EditorView,
        LikesView,
        FileViewer,
        templateStore
        ) {

        "use strict";
        var ReplyView = Backbone.View.extend({
            model: Reply,

            tagName: "div",

            className: "cp-annotation-reply",

            events: {
                "click .cp-annotation-edit": "editReply",
                "click .cp-annotation-delete": "deleteReply"
            },

            initialize: function(options) {
                this.listenTo(this.model, "change", this.render);
                this.listenTo(this.model, "remove", this.remove);
                this._annotationView = options.annotationView;
                this._parentView = options.commentView;
            },

            render: function() {
                if (!this.model.get("id")) {
                    return this;
                }

                this.$el.html(templateStore.get('Annotation.annotationReply')({
                    id: this.model.get("id"),
                    canEdit: this.model.get("hasEditPermission"),
                    canDelete: this.model.get("hasDeletePermission"),
                    author: this.model.get("author"),
                    comment: this.model.get("comment"),
                    date: this.model.get("date"),
                    resolved: this._parentView._isResolved()
                }));

                if (this.model.get("selected")) {
                    this.$el.addClass("selected");
                }
                else {
                    this.$el.removeClass("selected");
                }

                this.showLikes();
                // HACK: Trigger event that helps JIRA Issue Macro will handle asynchronous when edit comment
                AJS.trigger("ic-jim-async-supported");
                return this;
            },
            
            showLikes: function() {
                var likesView = new LikesView({
                    el: this.$el.find(".cp-annotation-reply-like"),
                    id: this.model.get('id'),
                    collection: this._parentView.model.likes,
                    _annotationView: this._annotationView
                });
                likesView.render();
            },
            
            editReply: function() {
                if(!this._annotationView.canOpenNewEditor()) {
                    return;
                }

                var content = this.model.get('editorFormat');
                this.editorView = this._annotationView._editorView = new EditorView({
                    editorSetup: function() {
                        var $loader = $(templateStore.get('Annotation.editorLoader')());
                        this.$el.find(".cp-annotation-comment").replaceWith($loader);
                        this.$el.find(".cp-annotation-actions").hide();
                    }.bind(this),
                    container: this.$el,
                    saveHandler: _.bind(this.saveReply, this),
                    cancelHandler: _.bind(this.restore, this),
                    postDeactivate: _.bind(this.restore, this),
                    content: content,
                    errorCallback: _.bind(this._handleEditorError, this),
                    restoreCallback: _.bind(this.restore, this),
                    mediaViewer: this._annotationView._mediaViewer
                });
                this.editorView.render();
            },

            saveReply: function(e) {
                e && e.preventDefault();
                if(this.editorView.getContent() === "") {
                    this._generateError(AJS.I18n.getText('cp.annotations.save.empty'));
                    return;
                }
                this._annotationView.clearErrorFlags();
                this.model.set({editorFormat: this.editorView.getContent()}, {silent: true});
                this.model.save(null, {
                    wait: true,
                    success: this.restore.bind(this),
                    error: this._handleError.bind(this)
                });
                this.editorView && this.editorView.remove();
            },

            restore: function(e) {
                //cause restore can also be passed a model as the first argument
                e && e.preventDefault && e.preventDefault();
                this.editorView && this.editorView.remove();
                this.render();
            },
            
            deleteReply: function() {
                var answer = window.confirm(AJS.I18n.getText('cp.annotations.delete.reply.confirmation.message'));
                if (answer) {
                    this.model.destroy({
                        wait: true,
                        error: this._handleError.bind(this)
                    });
                }
            },

            _handleError: function() {
                this._generateError(AJS.I18n.getText('cp.annotations.save.fail'));
            },

            _handleEditorError: function() {
                this._generateError(AJS.I18n.getText('cp.annotations.editor.open.fail'));
            },

            _generateError: function(error) {
                this._annotationView._generateError(error);
            }
        });

        return ReplyView;
    });

