define("cp/component/annotation/likes",
    [
        "backbone",
        "underscore",
        "jquery",
        "cp/component/annotation/like",
        "ajs"
    ],
    function(
        Backbone,
        _,
        $,
        Like,
        AJS
        ) {
        "use strict";

        /**
         * Likes collection rather than stores a collection of individual types, stores the likes for every comment/reply
         * in the annotation tree. Thus the structure of the likes collection looks like (in JSON):
         * {
         *      commentId: {
         *          likes: Array,
         *          content_type: "comment",
         *          ...
         *      },
         *
         *      commentId: {
         *          ...
         *      },
         * }
         */
        var Likes = Backbone.Collection.extend({
            model: Like,

            initialize: function(models, props) {
                this.contentId = props.contentId;
                this.replies = props.replies ? props.replies : [];

                this._skipFetch = false;
            },

            addLike: function(commentId, options) {
                var like = _.find(this.models, function(like) {
                    return like.get('content_id') === commentId;
                });

                if (!like) {
                    like = new Like();
                    //this is to trigger an add event for the view, the actual like will be added when the server
                    //responds in like.save
                    this.add(like);
                }
                like.save(null, {
                    id: commentId,
                    wait: true,
                    error: options.error
                });
            },

            removeLike: function(commentId, options) {
                var like = _.find(this.models, function(like) {
                    return like.get('content_id') === commentId;
                });

                //setting id so backbone will call delete, internally Backbone checks for the existence of an id
                //if no id exists, it assumes that it wasn't sync'd on the server and thus doesn't make a sync call
                //in our model. hacky, but this is a work around
                like.id = "";
                like.destroy({
                    id: commentId,
                    wait: true
                });
            },

            /**
             * @param method
             * @param collection
             * @param options {obj} if force = true, forces collection to fetch, even if it's already synced
             */
            sync: function(method, collection, options) {
                if(options.force) {
                    this._skipFetch = false;
                }

                if (this._skipFetch || !collection.contentId) {
                    return;
                }

                if (method === "read") {
                    this._skipFetch = true;
                    var ids = this._getCommentIds();
                    var dfd = $.Deferred();
                    $.ajax({
                        url: AJS.contextPath() + "/rest/likes/1.0/content/likes",
                        type: "POST",
                        data: JSON.stringify(ids),
                        contentType: "application/json"
                    }).then(function (data) {
                        var likes = _.map(data, function(like) {
                            return new Like(like);
                        });
                        collection.reset(likes);
                        dfd.resolve(likes);
                    }).fail(function (jqXHR, textStatus, errorThrown) {
                        this._skipFetch = false;
                        dfd.reject(jqXHR, textStatus, errorThrown);
                    }.bind(this));
                }
            },

            setReplies: function(collection) {
                this.replies = collection;
            },

            _getCommentIds: function() {
                var ids = [];
                //add our comment ID
                ids.push(this.contentId);
                if(this.replies) {
                    _.each(this.replies, function(reply) {
                        ids.push(reply.get('id'));
                    });
                }

                return {'ids' : ids};
            }
        });
        return Likes;
    });