define('cp/component/annotation/annotation-plugin', [
    'MediaViewer',
    'cp/component/annotation/annotation-view',
    'cp/component/annotation/annotation-button-view',
    'cp/component/annotation/pins-view',
    'cp/component/annotation/comments',
    'cp/component/annotation/comment',
    'cp/component/annotation/content-view',
    'underscore',
    'jquery',
    'ajs',
    'core/template-store-singleton'
], function (
    FileViewer,
    AnnotationView,
    AnnotationButtonView,
    PinsView,
    Comments,
    Comment,
    ContentView,
    _,
    $,
    AJS,
    templateStore
) {
    'use strict';
    var UnknownFileTypeView = FileViewer.require('viewers/unknown-file-type-view/unknown-file-type-view');

    var annotationPlugin = function (mediaViewer) {
        var config = mediaViewer.getConfig();
        if (!config.enableAnnotations || !config.commentService) { return; }

        var CommentService = config.commentService;

        ContentView.prototype.annotationOptions = {
            dropTarget: '.cp-annotatable'
        };

        mediaViewer.getView().fileSidebarView.addPanelView('annotations', AnnotationView);

        var controlsView = mediaViewer.getView().fileControlsView;
        var sidebarView = mediaViewer.getView().fileSidebarView;

        var shouldShowAnnotationButton = function() {
            var file = mediaViewer.getCurrentFile();
            return !file.get('isRemoteLink');
        };

        controlsView.addLayerView('annotationButton', AnnotationButtonView, {
            predicate: shouldShowAnnotationButton,
            weight: 5
        });

        var showAnnotationsPanel = annotationPlugin.showAnnotationsPanel;
        mediaViewer._fileState.on('cp.showAnnotations', _.partial(showAnnotationsPanel, mediaViewer));

        var enhanceFile = function(file) {
            if (file.createAnnotation) {
                return;
            }

            file.set("annotations", new Comments([], {service: new CommentService(file.get("id"), file.get("version")), fileModel: file}));

            if (FileViewer.isPluginEnabled('permalink')) {
                var permalinkPlugin = FileViewer.getPlugin('permalink');
                file.get("annotations").on("selected", function(annotation) {
                    if (annotation.get("id")) {
                        // Call the router so the permalink can be updated when permalink changes.
                        permalinkPlugin.setRouteForPin(file, annotation);
                    }
                    else {
                        // When saving a new pin the id of the annotation will not be set immediately.  Wait for the id
                        // to change then make the call to update the router.
                        annotation.on("change:id", function() {
                            permalinkPlugin.setRouteForPin(file, annotation);
                        });
                    }
                });
            }

            file.on("change:id", function () {
                this.get("annotations").service = new CommentService(file.get("id"), file.get("version"));
            });
            file.createAnnotation = function(props) {
                var attrs = _.extend(props, {fileModel: this});
                var annotations = this.get("annotations");
                var service = new CommentService(file.get("id"), file.get("version"));
                var annotation = new Comment(attrs, {service: service});
                annotations.add(annotation);
                return annotation;
            }.bind(file);
        };

        mediaViewer.getView().on("fv.fileChange", enhanceFile);

        var addShowResolvedFileAction = function(moreButton) {
            var file = mediaViewer.getCurrentFile();
            if (file.get("annotations").where({resolved: true}).length > 0) {
                moreButton.addFileAction({
                    key: "resolved.toggle",
                    text: AJS.I18n.getText('cp.show.resolved.feedback'),
                    callback: function () {
                        file.get("annotations").setFilter();
                        addHideResolvedFileAction(moreButton);
                    }.bind(this)
                });
            }
        };

        var addHideResolvedFileAction = function(moreButton) {
            var file = mediaViewer.getCurrentFile();
            moreButton.addFileAction({
                key: "resolved.toggle",
                text: AJS.I18n.getText('cp.hide.resolved.feedback'),
                callback: function() {
                    file.get("annotations").setFilter({resolved: false});
                    addShowResolvedFileAction(moreButton);
                }.bind(this)
            });
        };

        var updateActions = function(moreButton, collection) {
            var filter = collection.getFilter();
            if (_.isEqual(filter, {resolved: false})) {
                addShowResolvedFileAction(moreButton);
            } else {
                addHideResolvedFileAction(moreButton);
            }
        };
        var isDragging;

        mediaViewer.close = _.wrap(mediaViewer.close, function(close) {
            if (isDragging) { return; }
            close.apply(mediaViewer, Array.prototype.slice.call(arguments, 1));
        });

        mediaViewer.showFile = _.wrap(mediaViewer.showFile, function(showFile) {
            if (isDragging) { return; }
            return showFile.apply(mediaViewer, Array.prototype.slice.call(arguments, 1));
        });

        // Annotation layers are created asynchronously in mediaViewer.
        // It is not possible to determine if annotation layer is created or not by checking $(".annotationLayer").
        // Hence we need a flag isAnnotationLayerCreated to check if the annotation layer is already created.
        // It is expected that the annotation layer is only created once after a showFile().
        var isAnnotationLayerCreated = false;
        var initializeAnnotations = function (file) {
            var currentViewer = mediaViewer.getView().fileContentView.getLayerForName("content")._viewer;

            //we want to disable annotations if the current file is a web image or unknown file type
            //we have to add the unknown file type check here because the previewer doesn't know the rendered
            //file's actual type (after conversion) until after the file is actually loaded
            if(!shouldShowAnnotationButton() || currentViewer instanceof UnknownFileTypeView) {
                //we have to hide the annotation toolbar button because the toolbar loads before the file is converted
                if (currentViewer instanceof UnknownFileTypeView) {
                    // we only hide it for unknown file types because trying to hide the button for remote
                    // images throws an exception because the layer can't be found
                    controlsView.getLayerForName('annotationButton').$el.hide();
                }
                sidebarView.teardownPanel();
                return;
            }

            // Enable annotation button.
            var $fileControl = $('.cp-toolbar');
            var $annotateButton = $fileControl.find('#cp-file-control-annotate');

            //hack: if permission is undefined, the file fetch promise hasn't resolved so to be safe we render the annotations control
            if ($annotateButton.length === 0 && (file.get("hasReplyPermission") || file.get("hasReplyPermission") === undefined)) {
                $annotateButton = $(templateStore.get('Annotation.fileControlAnnotate')());
                // Enable chunky tooltip
                $.fn.tooltip && $annotateButton.tooltip({gravity: 's'});
                $annotateButton.appendTo($fileControl);
                $fileControl.css('margin-left', -$fileControl.width() / 2);
            }
            file.trigger("cp.control-added", {plugin: "annotation"});

            //add our annotation specific content view functions to the returned view
            currentViewer = _.extend(currentViewer, ContentView);
            if (!isAnnotationLayerCreated && !$('.cp-annotatable .cp-active-annotation').length) {
                currentViewer.renderAnnotations && currentViewer.renderAnnotations(PinsView);
                isAnnotationLayerCreated = true;
            }

            var moreButton = controlsView.getLayerForName('moreButton');

            var annotations = file.get("annotations");
            updateActions(moreButton, annotations);

            annotations.on("sync change:resolved filterUpdated", _.partial(updateActions, moreButton, annotations));
            annotations.fetchComments().done(function () {
                if (mediaViewer.getView().fileSidebarView.isPanelInitialized('annotations')) {
                    annotations.getCurrentOrNext();
                }
            });

            var dropSelector = currentViewer.annotationOptions.dropTarget;
            var annotationCreated = currentViewer.annotationOptions.annotationCreated;

            var onDrop = function (e, ui) {
                ui.helper.addClass("active");

                var $this = $(this);
                var offset = $this.offset();

                var relX = e.pageX - offset.left;
                var relY = e.pageY - offset.top;

                var x = relX / $this.width();
                var y = relY / $this.height();

                var annotation;
                if (annotationCreated) {
                    annotation = annotationCreated(this, e);
                }

                annotation = _.extend({}, annotation, {x: x, y: y});

                var leftPercentage = (annotation.x * 100) + "%";
                var topPercentage = (annotation.y * 100) + "%";

                $this.closest(dropSelector).append(ui.helper.detach());
                ui.helper.css({left: leftPercentage, top: topPercentage});


                mediaViewer._fileState.trigger("cp.showAnnotations");
                mediaViewer.getView().$el.find("#cp-file-control-annotate,#cp-control-panel-annotations").draggable("option", "disabled", true);

                sidebarView.getInitializedPanel('annotations').addAnnotation(annotation);
            };

            // Enable draggable annotation pin.
            var $annotateHandle = mediaViewer.getView().$el.find("#cp-file-control-annotate");
            var draggable = $annotateHandle.draggable({
                appendTo: mediaViewer.getView().el,
                helper: function () {
                    return $("<div class='annotation-pin'></div>");
                },
                cursor: "-webkit-grabbing",
                cursorAt: {left: 0, top: 0},
                scroll: false,
                stack: "img",
                revert: "invalid",
                start: function () {
                    isDragging = true;
                    currentViewer.autoToggleControls(false);
                    currentViewer.hideControls();
                    $(dropSelector).droppable({
                        tolerance: "pointer",
                        drop: onDrop
                    });
                },
                stop: function () {
                    isDragging = false;
                    currentViewer.showControls();
                    currentViewer.autoToggleControls(true);
                    $(dropSelector).droppable("destroy");
                }
            });

            var widget = draggable.data('draggable');
            if (widget && widget._mouseUp && widget._mouseMove
              && widget._trigger && widget._clear) {

                var clickEvent;

                var beginDrag = function (e) {
                    if (!isDragging && !widget.options.disabled) {
                        widget._mouseStart.call(widget, e);
                        isDragging = true;
                        clickEvent = e;
                        bindEvents();
                    }
                };

                var completeDrag = function (e) {
                    if ($(e.target).is($annotateButton)) { return; }
                    if (isDragging && !widget.options.disabled) {
                        widget._mouseDownEvent = clickEvent;
                        widget._mouseUp.call(widget, e);
                        isDragging = false;
                        unbindEvents();
                    }
                };

                var drag = function (e) {
                    if (isDragging && !widget.options.disabled) {
                        widget._mouseStarted = true;
                        widget._mouseMove.call(widget, e);
                    }
                };

                var cancelDrag = function (e) {
                    if (isDragging && !widget.options.disabled && e.keyCode === 27) {
                        widget._trigger("stop", e);
                        widget._clear();
                    }
                };

                var bindEvents = function() {
                    $(document).on("mousemove", drag);
                    $(dropSelector).on("click", completeDrag);
                    $(document).on("click", completeDrag);
                    $(document).on("keyup", cancelDrag);
                };

                var unbindEvents = function() {
                    $(document).off("mousemove", drag);
                    $(dropSelector).off("click", completeDrag);
                    $(document).off("click", completeDrag);
                    $(document).off("keyup", cancelDrag);
                };

                $annotateButton.click(beginDrag);
            }
        };

        var setupMode = function (file) {
            var $fileControl = $('.cp-toolbar');
            var $annotateButton = $fileControl.find('#cp-file-control-annotate');
            var $annotationLayer = $('.annotationLayer');
            var $annotations = $('.cp-active-annotation');

            if (mediaViewer.isInMode("PRESENTATION")) {
                $annotateButton.hide();
                $annotationLayer.hide();
                $annotations.hide();
            } else {
                initializeAnnotations(file);
                $annotateButton.show();
                $annotationLayer.show();
                $annotations.show();
            }
        };

        mediaViewer.on("fv.showFile", function(file) {
            isAnnotationLayerCreated = false;
            setupMode(file);
        });
        mediaViewer.on("fv.changeMode", function() {
            setupMode(mediaViewer.getCurrentFile());
        });
    };

    // External API
    annotationPlugin.showAnnotationsPanel = function (mediaViewer) {
        var sidebarView = mediaViewer.getView().fileSidebarView;
        if (!sidebarView.isPanelInitialized('annotations')) {
            if (sidebarView.isAnyPanelInitialized()) {
                sidebarView.teardownPanel();
            }
            sidebarView.initializePanel('annotations');
        }
        // HACK: Trigger event that helps JIRA Issue Macro will handle asynchronous first load panel
        AJS.trigger("ic-jim-async-supported");
    };

    return annotationPlugin;
});


(function(){
    var AnnotationPlugin = require('cp/component/annotation/annotation-plugin');
    var MediaViewer = require('MediaViewer');
    MediaViewer.registerPlugin('annotation', AnnotationPlugin);
})();
