define("cp/service/upload-service",
    [
        "ajs",
        "jquery",
        "plupload",
        "cp/service/files-service"
    ],
    function(
        AJS,
        $,
        plupload,
        FilesService
        ) {
        'use strict';

        var DEFAULT_MIME_TYPE = "application/x-upload-data";

        var contextPath = AJS.contextPath();

        var dragAndDropEntityId = AJS.Meta.get("drag-and-drop-entity-id"),
            atlToken = AJS.Meta.get("atl-token"),
            metaMaxFileSize = AJS.Meta.get("global-settings-attachment-max-size");

        /**
         * UploadService is used to upload new file version
         * @param previewingAttachment
         * @constructor
         */
        function UploadService(previewingAttachment) {
            this.previewingAttachment = previewingAttachment;
            this.filesService = new FilesService(previewingAttachment.get("ownerId"));
        }

        UploadService.prototype.getMetaMaxSize = function() {
            return metaMaxFileSize;
        };

        UploadService.prototype.getUploadUrl = function() {
            return contextPath + "/plugins/drag-and-drop/upload.action";
        };

        UploadService.prototype.buildUploadParams = function(file) {
            var params = {},
                extension = file.name.split('.').pop();

            params.pageId = this.previewingAttachment.get("ownerId");
            params.filename = this.previewingAttachment.get("title");
            params.size = file.size;

            if (dragAndDropEntityId) {
                params.dragAndDropEntityId = dragAndDropEntityId;
            }

            params.mimeType = plupload.mimeTypes[extension.toLowerCase()] || DEFAULT_MIME_TYPE;
            params.atl_token = atlToken;
            params.withEditorPlaceholder = false;

            return params;
        };

        UploadService.prototype.parseResponse = function(resp) {
            return $.parseJSON(resp).data;
        };

        UploadService.prototype.handleError = function(error) {
            var message = "";

            if (error.response) {
                try {
                    //only http errors
                    var result = $.parseJSON(error.response);
                    message = result.actionErrors[0];
                } catch (e) {
                    message = error.message;
                }
            } else {
                message = error.message;
                if (error.code === plupload.FILE_SIZE_ERROR) {
                    var fileName = error.file.name;
                    var fileSizeLimit = AJS.DragAndDropUtils.niceSize(metaMaxFileSize).toString();
                    message = AJS.I18n.getText("dnd.validation.file.too.large.withname", fileName, fileSizeLimit);
                }
                else if (error.code === plupload.FILE_EXTENSION_ERROR) {
                    message = AJS.I18n.getText("cp.validation.file.extension.error");
                }
            }

            return message;
        };

        UploadService.prototype.promiseFileModel = function(uploadResponse) {
            var latestVersionId = (this.previewingAttachment.getLatestVersion ? this.previewingAttachment.getLatestVersion() : this.previewingAttachment).get("id");
            return this.filesService.getFileWithId(latestVersionId);
        };

        UploadService.prototype.addVersionChangeComment = function(comment, attachment, successCallback, failedCallback) {
            var id = attachment.get("id");
            var url = contextPath + "/rest/api/content/" + attachment.get("ownerId") + "/child/attachment/" + id;
            var data = {id: id,
                version: {number: attachment.get("version") + 1},
                metadata: {comment: comment}};
            $.ajax({
                url: url,
                type: "PUT",
                data: JSON.stringify(data),
                dataType: "json",
                contentType: "application/json; charset=utf-8"}
            ).done(successCallback)
            .fail(function(data){
                failedCallback(JSON.parse(data.responseText).message);
            });
        };

        return UploadService;
    }
);