define("cp/confluence/file-preview-loader",
    [
        "jquery",
        "underscore",
        "ajs",
        "cp/confluence/preview-support",
        "cp/confluence/async-module-backend",
        "MediaViewer",
        "cp/confluence/conversion-poller",
        "cp/service/files-service",
        "cp/service/comments-service",
        "cp/service/versions-service",
        "cp/confluence/file-utils",
        "confluence/api/event"
    ],
    function(
        $,
        _,
        AJS,
        PreviewSupport,
        asyncModuleBackend,
        MediaViewer,
        ConversionPoller,
        FilesService,
        CommentsService,
        VersionsService,
        fileUtils,
        event
    ) {
            "use strict";

            var mediaViewer,
                currentViewMode,
                attachmentsFetched = false;

            var analyticsBackendAJS = function (key, data) {
                AJS.trigger('analyticsEvent', {
                    name: key,
                    data: data
                });
            };

            //import dependencies from MediaViewer
            var fileTypes = MediaViewer.require('file-types/file-types');

            function getFileSrc($file) {
                return $file.attr("data-image-src") || $file.attr("data-file-src") || $file.attr("src") || $file.attr("href");
            }

            function getContentType($file) {
                return $file.attr("data-linked-resource-content-type") || $file.attr("data-mime-type") || "image/web";
            }

            function buildFileFromHtmlObject(file) {
                var $file = $(file);
                var fileSrc = getFileSrc($file);
                var fileName = $file.attr("data-linked-resource-default-alias");
                return {
                    src: fileSrc,
                    srcDownload: fileSrc + "&download=true",
                    thumbnail: fileSrc,
                    type: getContentType($file),
                    title: fileName || fileSrc,
                    name: fileName,
                    id: $file.attr("data-linked-resource-id"),
                    version: $file.attr("data-linked-resource-version"),
                    ownerId: $file.attr("data-linked-resource-container-id"),
                    isRemoteLink: $file.hasClass("confluence-external-resource")
                };
            }

            function makeFileArray($files) {
                return _.map($files, function (file) {
                    return buildFileFromHtmlObject(file);
                });
            }

            function autoShowAnnotationsPanel(mediaViewer) {
                if (MediaViewer.isPluginEnabled('annotation')) {
                    mediaViewer.once("fv.showFile", function () {
                        MediaViewer.getPlugin('annotation').showAnnotationsPanel(mediaViewer);
                    });
                }
            }

            function showPreviewer(fileList, file, viewMode, shouldAutoShowAnnotations) {

                var filesPromise;

                if (mediaViewer && attachmentsFetched && currentViewMode === viewMode) {
                    createOrUpdateFileView(fileList, viewMode, true);
                    filesPromise = $.when();
                } else {
                    // User switched to a different mode.
                    // IN HERE
                    mediaViewer && mediaViewer.close();
                    filesPromise = setupPreviewerAndShow(viewMode);
                    attachmentsFetched = true;
                }

                filesPromise.then(function() {
                    if (!file) {
                        return;
                    }

                    // We update the files again once the rest call to retrieve attachments/files has returned.
                    //fileViewer.updateFiles(deDupeFiles(fileList), matchId);

                    var $file = $(file);

                    var id = $file.attr("data-linked-resource-id"),
                            ownerId = $file.attr("data-linked-resource-container-id"),
                            src = $file.attr("data-image-src") || $file.attr("data-file-src") || $file.attr("src");

                    var query = (id && ownerId) ? { id: id, ownerId: ownerId} : { src: src };

                    mediaViewer.open(query, 'main');

                    if (shouldAutoShowAnnotations) {
                        autoShowAnnotationsPanel(mediaViewer);
                    }
                })
            }

            var commentSection = {

                showPreviewerForComment: function (file, $commentContainer, viewMode, shouldAutoShowAnnotations) {
                    var $file = $(file);
                    var $fileInComments = $commentContainer.find(PreviewSupport.getFileSelectorString());
                    var $files = _.uniq($fileInComments, function($file) {
                        return $file.src || $file.href;
                    });
                    var fileList = makeFileArray($files);

                    commentSection.setupPreviewerForComment(fileList, viewMode, false).then(function() {
                        var id = $file.attr("data-linked-resource-id"),
                            ownerId = $file.attr("data-linked-resource-container-id"),
                            src = $file.attr("data-image-src") || $file.attr("data-file-src") || $file.attr("src");

                        var query = (id && ownerId) ? { id: id, ownerId: ownerId } : { src: src };
                        mediaViewer.open(query, 'comments');

                        if (shouldAutoShowAnnotations) {
                            autoShowAnnotationsPanel(mediaViewer);
                        }
                    })
                },

                setupPreviewerForComment: function (fileList, viewMode, enablePermalinks) {
                    var filesPromise = $.when();

                    var pageId = AJS.Meta.get('page-id');
                    if (pageId) {
                        // We must have a page ID otherwise service call will fail.
                        filesPromise = loadFiles(fileList, pageId, false);
                    }

                    createOrUpdateFileView(fileList, viewMode, enablePermalinks);

                    filesPromise.then(function(files) {
                        // We update the files again once the rest call to retrieve attachments/files has returned.
                        mediaViewer.updateFiles(files, fileUtils.matchId);
                    });

                    return filesPromise;
                }
            };

            /**
             * Show previewer for one single file. In other words, there is no mini-mode in simple previewer.
             */
            function showPreviewerForSingleFile(file, viewMode, source) {
                var mediaViewer = setupPreviewForSingleFile(file, viewMode);
                mediaViewer.open({id: mediaViewer.getFiles()[0].id}, source);
            }

            /**
             * Set up Preview for single file view. It includes:
             *      No mini-mode.
             *      No permalink support.
             *
             * @returns MediaViewer
             */
            function setupPreviewForSingleFile(file, viewMode) {
                var files = _.isArray(file) ? file : makeFileArray($(file));
                createOrUpdateFileView(files, viewMode, false);

                return mediaViewer;
            }

            var _poller;

            /**
             * Currently, Confluence provides HD url for ALL document as a workaround for CONFDEV-40641,
             *  this function however will set hdURL to null if a document does not actually have an HD version
             * @param file a object of mediaviewer
             * @returns a promise of an appropriate override to use in rendering mediaviewer
             */
            var getHDOverrides = function (file) {
                var overrideDeferred = $.Deferred();
                var noOverride = {};
                var removeHdOverride = {
                    src_hd: null,
                    poster_hd: null
                };
                if (file.get("src_hd")) {
                    $.ajax({
                        type: "HEAD",
                        async: true,
                        url: file.get("src_hd")
                    }).then(function (data, textStatus, xhr) {
                        overrideDeferred.resolve(xhr.status !== 200 ? removeHdOverride : noOverride);
                    }, function (jqXHR, textStatus, errorThrown) {
                        // To be super safe, we remove HD if cannot check its existence
                        overrideDeferred.resolve(removeHdOverride);
                    });
                } else {
                    overrideDeferred.resolve(noOverride);
                }
                return overrideDeferred.promise();
            };

            var tokenFetcher = function (file) {
                var dfd = new $.Deferred();
                dfd.resolve({
                    src: file.get('src'),
                    srcDownload: file.get('srcDownload')
                });
                return dfd.promise();
            };

            var isPreviewGenerated = function (file) {
                _poller && _poller.stop();

                var attachmentId = file.get("id");
                var version = file.get("version");
                var dfd = $.Deferred();

                if (!attachmentId) { return $.when(true, file.get("src"), file.get("type")); }

                _poller = new ConversionPoller(attachmentId, version, "conversion", function (status) {
                    if (status === "converting" || status === "busy") {
                        dfd.resolve(false, file.get("src"), file.get("type"));
                    }
                });

                _poller.promise()
                .then(function (previewSrc, previewType) {
                    if (dfd.state() !== 'pending') { return; }

                    getHDOverrides(file).done(function(overrideParam) {
                        dfd.resolve(true, previewSrc, previewType, overrideParam);
                    }).fail(function() {
                        dfd.resolve(true, previewSrc, previewType);
                    });
                })
                .fail(function (reason) {
                    if (dfd.state() !== 'pending') { return; }
                    if (reason === "cancelled") {
                        dfd.reject(reason);
                    } else {
                        getHDOverrides(file).done(function(overrideParam) {
                            dfd.resolve(true, file.get("src"), file.get("type"), overrideParam);
                        }).fail(function() {
                            dfd.resolve(true, file.get("src"), file.get("type"));
                        });
                    }
                });

                return dfd.promise();
            };

            var generatePreview = function (file) {
                _poller && _poller.stop();

                var attachmentId = file.get("id");
                var version = file.get("version");
                var dfd = $.Deferred();
                _poller = new ConversionPoller(attachmentId, version, "conversion");
                _poller.promise()
                    .done(function (previewSrc, previewType) {
                        dfd.resolve(previewSrc, previewType);
                    }).fail(function (reason) {
                        if (reason !== "cancelled") {
                            dfd.resolve(file.get("src"), file.get("type"));
                        } else {
                            dfd.reject(reason);
                        }
                    });

                return dfd.promise();
            };

            var generateThumbnail = function (file) {
                var type = file.get("type"),
                    src = file.get("src"),
                    attachmentId = file.get("id"),
                    isRemoteLink = file.get("isRemoteLink"),
                    dfd = $.Deferred();


                /* Even though we might know how to display img files, they should still go through the conversion to
                 * a thumbnail. We can't store files locally in Vertigo. */
                var dclImageThumbnails = AJS.DarkFeatures.isEnabled("previews.dcl-image-thumbnails");

                if ((!dclImageThumbnails && !isRemoteLink && fileTypes.isImageBrowserSupported(type))) {
                    var thumbnail = src.replace('/attachments/', '/thumbnails/');
                    dfd.resolve(thumbnail, "image/jpeg");
                } else if (attachmentId) {
                    var version = file.get("version");
                    var poller = new ConversionPoller(attachmentId, version, "thumbnail");
                    poller.promise().done(function (previewSrc, previewType) {
                        dfd.resolve(previewSrc, previewType);
                    }).fail(function () {
                        dfd.reject();
                    });
                } else {
                    dfd.resolve(src, type);
                }
                return dfd.promise();
            };

            function setupPreviewer(files, viewMode) {
                currentViewMode = viewMode;

                var staticPrefix = AJS.Meta.get("static-resource-url-prefix");
                var webResource = 'com.atlassian.confluence.plugins.confluence-previews:mediaviewer-chunks';
                var mediaviewerBasePath = staticPrefix + '/download/resources/' + webResource + '/';

                // Default FileViewer config for full mode.
                var configs = {
                    appendTo: $('body'),
                    files: files,
                    filesService: FilesService,
                    commentService: CommentsService,
                    versionsService: VersionsService,
                    moduleBackend: asyncModuleBackend,
                    enableAnnotations: AJS.DarkFeatures.isEnabled('file-annotations'),
                    enablePermalinks: AJS.DarkFeatures.isEnabled('previews.sharing'),
                    enableMiniMode: true,
                    enableShareButton: true,
                    enableVersionNavigation: true,
                    viewers: ['image', 'document', 'video'],
                    analyticsBackend: analyticsBackendAJS,
                    assets: {
                        basePath: mediaviewerBasePath
                    },
                    i18n: AJS.I18n.keys,
                    fetchToken: tokenFetcher
                };
                if (viewMode === PreviewSupport.VIEW_MODE.SIMPLE) {
                    configs.enableAnnotations = false;
                    configs.enableMiniMode = false;
                    configs.enableShareButton = false;
                    configs.enableVersionNavigation = false;
                }

                if (AJS.DarkFeatures.isEnabled("previews.conversion-service")) {
                    configs.isPreviewGenerated = isPreviewGenerated;
                    configs.generatePreview = generatePreview;
                    configs.generateThumbnail = generateThumbnail;
                }

                mediaViewer = new MediaViewer(configs);

                // Propagate the mediaviewer events to AJS to be consumed in confluence-browser-metrics/viewcontent
                mediaViewer.on("fv.showFile", function() {
                    event.trigger("confluence-previews.fileviewer.completed");
                });

                //disable key bindings in the previewer
                var fileView = mediaViewer.getView();

                fileView.show = _.compose(fileView.show, function (e) {
                    AJS.trigger("remove-bindings.keyboardshortcuts");
                    return e;
                });

                fileView.hide = _.compose(fileView.hide, function (e) {
                    AJS.trigger("add-bindings.keyboardshortcuts");
                    return e;
                });

                return mediaViewer;
            }

            function createOrUpdateFileView(files, viewMode, enablePermalinks) {
                if (!mediaViewer ||
                    currentViewMode !== viewMode) { // User switched to a different mode.
                    // IN HERE
                    mediaViewer && mediaViewer.close();
                    setupPreviewer(files, viewMode);
                }
                mediaViewer.updateFiles(fileUtils.deDupeFiles(files), fileUtils.matchId);
                if (MediaViewer.isPluginEnabled("permalink")) {
                    MediaViewer.getPlugin("permalink").setRoutingEnabled(enablePermalinks);
                }
            }

            function setupPreviewerAndShow(viewMode) {
                var $files = _.uniq($(PreviewSupport.getFileSelectorString()), function(file) {
                    return $(file).attr("data-linked-resource-id") || file.src;
                });
                var fileList = makeFileArray($files);

                // Set the user timezone so that pretty date formatting works.
                var timezoneOffset = AJS.Meta.get('user-timezone-offset');
                if (!timezoneOffset)  {
                    AJS.Meta.set('user-timezone-offset', 0);
                }

                var filesPromise;

                var pageId = AJS.Meta.get('page-id');
                if (pageId) {
                    // We must have a page ID otherwise service call will fail.
                    filesPromise = loadFiles(fileList, pageId, true);
                } else {
                    // No page ID means we can only display what was visible on the page.  Return a dummy promise for
                    // files rest call.
                    filesPromise = $.when();
                }

                return filesPromise.then(function(files) {
                    createOrUpdateFileView(files, viewMode, true);
                    if (MediaViewer.isPluginEnabled("permalink")) {
                        MediaViewer.getPlugin("permalink").startRouting();
                    }
                });
            }

            /**
             * Load files via the FilesService
             *
             * @param fileList
             * @param pageId the current page id
             * @param loadHiddenFiles a boolean flag indicates if we want to load other files in the page
             */
            function loadFiles(fileList, pageId, loadHiddenFiles) {
                // Fetch the attachments from the resource and update files
                var attachedImages = _.filter(fileList, function (file) {return !!file.id;}),
                    attachedImageIds = _.pluck(attachedImages, 'id'),
                    contentId = pageId,
                    service = new FilesService(contentId);

                //if fileList is not set (because there are no images in view mode), then retrieve all attached files
                var promises = [];
                if(attachedImageIds.length) {
                    promises.push(service.getFilesWithId(attachedImageIds));

                    //if we're in comment view mode, we don't want to fetch any other file besides the one we're viewing
                    if (loadHiddenFiles) {
                        promises.push(service.getFilesWithoutId(attachedImageIds));
                    }
                } else {
                    promises.push(service.getFiles());
                }

                return $.when.apply($, promises).pipe(function () {
                    var attachedFiles = _.reduce(arguments, function (a, b) { return a.concat(b); });

                    var isTriggerAllFileTypes = AJS.DarkFeatures.isEnabled("previews.trigger-all-file-types");
                    var isConversionServiceEnabled = AJS.DarkFeatures.isEnabled("previews.conversion-service");

                    var filteredFiles = attachedFiles;
                    if (!isTriggerAllFileTypes && !isConversionServiceEnabled) {
                        filteredFiles = _.filter(attachedFiles, function (file) {
                            return fileTypes.isImage(file.type) ||
                                    (fileTypes.isPDF(file.type) && PreviewSupport.isPDFSupported());
                        });
                    }

                    return fileUtils.mergeFiles(fileList, filteredFiles);
                });
            }

            return {
                showPreviewer: showPreviewer,
                showPreviewerForSingleFile: showPreviewerForSingleFile,
                setupPreviewForSingleFile: setupPreviewForSingleFile,
                showPreviewerForComment: commentSection.showPreviewerForComment
            };
    });
