define("cp/component/upload/attachment-uploader",
    [
        "jquery",
        "plupload",
        "ajs"
    ],
    function (
        $,
        plupload,
        AJS
    ) {
        "use strict";

        var BatchProgress = function () {
            this._workIdToBytesUploaded = {};
            this._totalBytes = 0;
        };

        BatchProgress.prototype = {
            update: function (workId, uploaded, fileSize) {
                if (!(workId in this._workIdToBytesUploaded)) {
                    this._totalBytes += fileSize;
                }
                this._workIdToBytesUploaded[workId] = uploaded;
            },
            percentComplete: function () {
                var totalBytesUploaded = 0;
                $.each(this._workIdToBytesUploaded, function (key, value) {
                    totalBytesUploaded += value;
                });
                return Math.round(totalBytesUploaded * 100 / this._totalBytes);
            }
        };

        /**
         * This class handles and forwards upload events from plupload to UploadNewVersionDialogView.
         */
        var AttachmentUploader = function ($browserButton, $container, attachment, uploadService) {
            /**
             * This class is used to public methods to be called from UploadNewVersionDialogView to interact with plupload
             */
            var Uploader = Backbone.Model.extend({
                initialize: function () {
                    $(window).on('resize.attachmentUploader', function () {
                        if (plUploaderInstance) {
                            plUploaderInstance.refresh();
                        }
                    });
                },
                cancelUpload: function () {
                    currentUploadingFile && plUploaderInstance && plUploaderInstance.removeFile(currentUploadingFile);
                    currentUploadingFile = null;
                },
                destroy: function () {
                    $(window).off('resize.attachmentUploader');

                    if (plUploaderInstance) {
                        plUploaderInstance.destroy();
                    }
                }
            });

            var currentUploadingFile = null,
                batchProgress = null,
                uploader = new Uploader();

            function _createPlupload() {
                var uploader = new plupload.Uploader({
                    runtimes: 'html5',
                    dragdrop: false,
                    browse_button: $browserButton.id,
                    multipart: false,
                    stop_propagation: true,
                    max_file_size: parseInt(uploadService.getMetaMaxSize(), 10),
                    inputFileClazz: "file-preview-input-file",
                    filters: attachment ? _getFilters(attachment.get("title")) : null,
                    multi_selection: false,
                    container: null
                });

                var uploadHandler = new UploadHandler();

                uploader.init();

                uploader.bind('Started', uploadHandler.handleStarted);
                uploader.bind('FilesAdded', uploadHandler.handleFilesAdded);
                uploader.bind("BeforeUpload", uploadHandler.handleBeforeUpload);
                uploader.bind("UploadProgress", uploadHandler.handleUploadProgress);
                uploader.bind('FileUploaded', uploadHandler.handleFileUploaded);
                uploader.bind("Error", uploadHandler.handleError);
                uploader.bind("UploadComplete", uploadHandler.handleUploadComplete);

                return uploader;
            }

            /**
             * Get filters from current previewing file
             * e.g. "image.jpg" is being previewed => return filters: [{title: "filter", extensions: "jpg,jpeg,jpe"}]
             * @param attachmentName
             * @returns {Array}
             * @private
             */
            function _getFilters(attachmentName) {
                var filters = [];
                var ext = (attachmentName.indexOf(".") !== -1) ? attachmentName.split(".").pop() : null;
                if (ext) {
                    var mime = plupload.mimeTypes[ext.toLowerCase()];
                    var extensions = (mime) ? plupload.mineTypeToExtensionsMap[mime] : [ext];
                    filters.push({title: "filter", extensions: extensions.join(",")});
                }

                return filters;
            }

            function UploadHandler() {
            }

            UploadHandler.prototype.handleStarted = function () {
                uploader.trigger("cp.uploader.uploadStarted");
            };

            UploadHandler.prototype.handleFilesAdded = function (up, files) {
                uploader.trigger("cp.uploader.filesAdded", files[0]);
                up.start();
            };

            UploadHandler.prototype.handleBeforeUpload = function (up, file) {
                batchProgress = new BatchProgress();
                currentUploadingFile = file;

                var url = uploadService.getUploadUrl(),
                    params = uploadService.buildUploadParams(file);

                up.settings.url = plupload.buildUrl(url, params);
            };

            UploadHandler.prototype.handleUploadProgress = function (up, file) {
                batchProgress.update(file.id, file.loaded, file.size);
                var percent = batchProgress.percentComplete() / 100;

                uploader.trigger("cp.uploader.uploadProgress", percent);
            };

            UploadHandler.prototype.handleFileUploaded = function (up, file, result) {
                //we need to do this check because when we cancel uploading file,
                //after remove that file from uploading queue, plupload still fires "FileUploaded" event
                if (up.getFile(file.id)) {
                    uploader.trigger("cp.uploader.fileUploaded", file, result);
                }
            };

            UploadHandler.prototype.handleError = function (up, error) {
                uploader.trigger("cp.uploader.error", uploadService.handleError(error));
                currentUploadingFile = null;
            };

            UploadHandler.prototype.handleUploadComplete = function (up, error) {
                batchProgress = null;
                currentUploadingFile = null;
            };

            var plUploaderInstance = _createPlupload();

            return uploader;
        };

        return AttachmentUploader;
});