/**
 * Embeds the "Edit" button directly into any appropriate placeholder
 * @since 6.14
 */
define('cp/component/companion/embedded-edit-with',[
    'jquery',
    'ajs',
    'backbone',
    'document',
    'cp/component/companion/companion-view',
    'cp/service/files-service',
    'confluence/api/event',
    'confluence/api/logger',
], function (
    $,
    AJS,
    Backbone,
    document,
    CompanionMainView,
    FilesService,
    Event,
    logger
) {
    'use strict';

    var embeddedEditButtons = [];
    var placeholders = [];

    /**
     * Retrieves all edit button placeholder elements on the page and inserts the function edit button (CompanionMainView)
     * within them.
     *
     * @returns {Array} Array of promises that resolve to the CompanionMainView objects that we embedded on page load
     */
    var bindEmbeddedEdit = function() {
        var allEditButtonPlaceholders = $('.companion-edit-button-placeholder[data-linked-resource-id][data-linked-resource-container-id]');
        var loadEditButtonPromise;
        allEditButtonPlaceholders.each(function () {
            loadEditButtonPromise = loadEditButton(this)['catch'](function () {
                // Ignore duplicate embedding errors
                return Promise.resolve(undefined)
            });
            embeddedEditButtons.push(loadEditButtonPromise);
        });
        // WARNING: External JS can call "loadEditButton" directly and after page load. Therefore these CompanionMainView
        // objects may not have been added to "embeddedEditButtons" and may not be included in the resolved array.
        Event.trigger('cp-companion-embedded-edit-buttons-loaded', getCompanionMainViewObjects());

        return embeddedEditButtons;
    };


    /**
     * Embeds CompanionMainView into a particular placeholder element.
     *
     * @param placeholder DOM element to insert CompanionMainView within
     * @returns {Promise} Resolves the CompanionMainView object that was inserted. Rejects when the placeholder is full.
     */
    var loadEditButton = function (placeholder) {
        var $placeholder = $(placeholder);

        // Ensure that we do not embed the edit button view (CompanionMainView) into a placeholder more than once
        var injectedButton = $placeholder.children('.companion-edit-button');
        var placeholderEmbeddedIndex = placeholders.indexOf($placeholder[0]);
        if (injectedButton.length > 0 || placeholderEmbeddedIndex !== -1) {
            return Promise.reject('EmbeddedEditWith.loadEditButton: Companion Edit With button already loaded');
        }
        placeholders.push($placeholder[0]);

        var buttonContext = {
            pageID: $placeholder.attr("data-linked-resource-container-id"),
            attachmentID: $placeholder.attr("data-linked-resource-id"),
            templateName: $placeholder.attr("data-template-name"),
            sourceLocation: $placeholder.attr("data-source-location")
        };
        var _loadEditButtonInternalPromise = _loadEditButtonInternal(buttonContext, placeholder);
        return _loadEditButtonInternalPromise;
    };

    var _loadEditButtonInternal = function(buttonContext, placeholder, CompanionMainViewClass) {
        var fileService = new FilesService((buttonContext.pageID));

        var loadEditButtonPromise = new Promise(function(resolve, reject) {
            fileService.getFileWithId(buttonContext.attachmentID).then(function(file) {
                // Avoid MediaViewer dependency, see 'core/file' (file object is a simple backbone model)
                var fileModel = new Backbone.Model(file);
                try {
                    var companionMainView = new CompanionMainView({template: buttonContext.templateName, el: placeholder, file: fileModel, sourceLocation: buttonContext.sourceLocation});
                    companionMainView.render();
                    resolve(companionMainView);
                } catch (e) {
                    reject(e);
                }
            });
        });

        return loadEditButtonPromise;
    };


    /**
     * Returns all of the "CompanionMainView" Backbone View objects that have been embedded into the page at page load.
     * The CompanionMainView provides the edit functionality.
     *
     * @returns {Promise} Resolves an array of CompanionMainView objects.
     */
    var getCompanionMainViewObjects = function () {
        // Return all companionMainView objects that are embedded on the page. Ignore promises that rejected (duplicate insertions)
        return Promise.all(embeddedEditButtons).then(function(results) {
            return results.filter(function(companionMainView) {
                return companionMainView !== undefined;
            })
        });
    };

    return {
        _init: bindEmbeddedEdit,
        loadEditButton: loadEditButton
    };
});

AJS.toInit(function() {
    try {
        require('cp/component/companion/embedded-edit-with')._init();
    } catch (e) {
        console.warn("Couldn't load edit resources", e)
    }
});
