define("cp/component/annotation/replies",
    [
        "backbone",
        "underscore",
        "cp/component/annotation/reply"
    ],
    function(
        Backbone,
        _,
        Reply
        ) {
        "use strict";

        var Replies = Backbone.Collection.extend({

            initialize: function(attrs, opts) {
                this.parentModel = opts.parentModel;
                this.service = opts && opts.service;
                //external variable that we can use to check if our collection is synced
                this.synced = false;
                //internal variable to cache our data and skip multiple duplicate network requests
                this._skipFetch = false;
            },

            /**
             * @param method
             * @param collection
             * @param options {obj} if force = true, forces collection to fetch, even if it's already synced
             */
            sync: function(method, collection, options) {
                var id = collection.parentModel && collection.parentModel.get("id");
                if (options.force) {
                    this._skipFetch = false;
                }

                if (!this.service || !id || this._skipFetch) {
                    return;
                }

                if (method === "read") {
                    this._skipFetch = true;
                    var promise =  collection.service.getReplies({
                            parentId: id
                        }),
                        that = this;

                    promise.done(function(replies) {
                        replies = _.map(replies, function(reply) {
                            return new Reply(reply, {service: that.service});
                        });
                        this.synced = true;
                        collection.reset(replies);
                    }.bind(this)).fail(function() {
                        this.synced = false;
                        this._skipFetch = false;
                    }.bind(this));
                }
            },

            addReply: function(props, options) {
                var reply = new Reply(props, {service: this.service});
                this.setSelected(reply);
                this.add(reply);
                reply.save(null, {
                    wait: true,
                    error: options.error,
                    success: options.success
                });
            },

            selectReplyWithId: function(replyId) {
                if (!this.isSynced()) {
                    this.listenToOnce(this, "reset", function () {
                        this.setSelected(this.findWhere({id: replyId}));
                    });
                } else {
                    this.setSelected(this.findWhere({id: replyId}));
                }
            },

            setSelected: function(model) {
                this.invoke('set', {selected: false});
                if (model) {
                    model.set('selected', true);
                }
            },

            isSynced: function() {
                return this.synced;
            }
        });

        return Replies;

    });