define(
    "cp/component/annotation/annotation-view",
    [
        "backbone",
        "jquery",
        "ajs",
        "cp/component/annotation/annotation-header-view",
        "cp/component/annotation/comment-view",
        "cp/component/annotation/add-annotation-view",
        "core/template-store-singleton"
    ],
    function(
        Backbone,
        $,
        AJS,
        AnnotationHeaderView,
        CommentView,
        AddAnnotationView,
        templateStore
     ) {
        "use strict";
        var AnnotationView = Backbone.View.extend({

            id: "cp-annotations",

            tagName: "div",

            initialize: function(options) {
                this._mediaViewer = options.mediaViewer;
                this.fileSidebarView = options.panelView;

                this.model = this._mediaViewer.getCurrentFile();
                this.annotations = this.model.get("annotations");
                this.listenTo(this.annotations, "reset sync selected unselected add remove", this.render);
                this.listenTo(this.annotations, 'pinSelected', this.show);
                this.listenTo(this.annotations, 'filterUpdated', this.updateWithFiltered);
                this.listenTo(this._mediaViewer, 'cp.close-editor', this._closeEditor);

                this.show();
                this.render();
            },

            teardown: function () {
                this.hide();
                this.commentView && this.commentView.off().remove();
                this.addAnnotationView && this.addAnnotationView.teardown() && this.addAnnotationView.off().remove();
            },

            hide: function() {
                $("#cp-control-panel-annotations").removeClass("focused");

                this.annotations.setSelected();

                this._mediaViewer._fileState.trigger("cp.hideAnnotations");
            },

            show: function() {
                $("#cp-control-panel-annotations").addClass("focused");
            },

            addAnnotation: function(position) {
                if(!this.canOpenNewEditor()) {
                    $("#cp-file-control-annotate").draggable("option", "disabled", false);
                    return;
                }
                this.show();

                this.annotations.setSelected();

                var annotation = this._mediaViewer.getCurrentFile().createAnnotation({
                    author: this.currentUser,
                    position: [position.x, position.y],
                    pageNumber: position.pageNumber,
                    isNew: true
                });

                this.annotations.setSelected(annotation, true);

                this.addAnnotationView = new AddAnnotationView({
                    annotation: annotation,
                    collection: this.annotations,
                    mediaViewer: this._mediaViewer,
                    annotationView: this
                });
                this.$el.html(this.addAnnotationView.$el);
                this.addAnnotationView.render();

                this.$el.find("textarea.cp-add-annotation").focus();
            },

            render: function() {
                if (this.annotations) {
                    var current = this.annotations.getCurrent();

                    if (!current) {
                        this.$el.html(templateStore.get('Annotation.blankAnnotation')());
                    } else {
                        this.headerView = new AnnotationHeaderView({model: this.model, annotationView: this});
                        this.commentView = new CommentView({model: current, annotationView: this});
                        this.$el
                            .empty()
                            .append(this.headerView.render().$el)
                            .append(this.commentView.render().el);
                    }
                }

                // HACK: Trigger event that helps JIRA Issue Macro will handle asynchronous
                AJS.trigger("ic-jim-async-supported");
                return this;
            },

            _generateError: function(error) {
                this.headerView._generateError(error);
            },

            //helper function to check if we currently have an editor open, and if we have permission to close it
            canOpenNewEditor: function() {
                return !this._editorView || this._editorView.checkIfOpen();
            },

            isCommentVisible: function(model) {
                return this.annotations.isFiltered(model);
            },

            clearErrorFlags: function() {
                this.headerView._clearError();
            },
            
            _closeEditor: function() {
                this._editorView && this._editorView.remove();
            },

            updateWithFiltered: function() {
                this.annotations.getCurrentOrNext();
                this.render();
            }
        });

        return AnnotationView;
    });