define('confluence/page-hierarchy/service/dialog-service', [
    'ajs',
    'confluence/legacy',
    'confluence/page-hierarchy/util/loading-indicator',
    'confluence/page-hierarchy/util/execute-if-function'
], function (AJS,
             Confluence,
             LoadingIndicator,
             executeIfFunction) {
    var HINT_SELECTOR = '#hint';
    var SPINNER_SELECTOR = '.button-spinner';
    // Stores our dialog references
    var dialogs = {};

    /**
     * Bind function that adds show method to a dialog
     * @param {string} name        unique name of the dialog
     * @param {object} options     options trickling down from get method
     * @returns {function} See usage
     * @private
     */
    function _show(name, options) {
        return function () {
            _ensureElement(name, options);
            var $element = dialogs[name].$element;
            var show = function () {
                dialogs[name].showTime = new Date().getTime();
                AJS.dialog2(_selector(name, options)).show();
            };
            if (options.showDelay) {
                dialogs[name].showDelay = setTimeout(function () {
                    show();
                }, options.showDelay);
            } else {
                show();
            }

            // Show method needs to be delayed because otherwise it will run before the init method which is not good.
            setTimeout(function () {
                executeIfFunction(options.onShow, $element);
            }, 0);
        };
    }

    /**
     * Bind function that adds hide method to a dialog
     * @param {string} name        unique name of the dialog
     * @param {object} options     options trickling down from get method
     * @returns {function} See usage
     * @private
     */
    function _hide(name, options) {
        return function () {
            if (!dialogs[name]) {
                return;
            }
            var $element = dialogs[name].$element;
            if (dialogs[name].showDelay) {
                clearTimeout(dialogs[name].showDelay);
            } else {
                var hide = function () {
                    if ($element && $element.is(':visible')) {
                        AJS.dialog2(_selector(name, options)).hide();
                    }
                };
                _minimumShowDuration(name, options.minimumShowDuration || 0, hide);
            }

            executeIfFunction(options.onHide, $element);
        }
    }

    /**
     * Bind function that adds remove method to a dialog
     * @param {string} name        unique name of the dialog
     * @param {object} options     options trickling down from get method
     * @returns {function} See usage
     * @private
     */
    function _remove(name, options) {
        return function () {
            if (!dialogs[name]) {
                return;
            }
            var $element = dialogs[name].$element;
            if ($element) {
                if (dialogs[name].showDelay) {
                    clearTimeout(dialogs[name].showDelay);
                }
                var remove = function () {
                    executeIfFunction(options.onRemove, $element);
                    AJS.dialog2(_selector(name, options)).remove();
                };
                _minimumShowDuration(name, options.minimumShowDuration || 0, remove);
            }
            dialogs[name].$element = undefined;
        }
    }

    /**
     * Bind function that adds async method to a dialog to bundle the loadingIndicator
     * and hide those details from the dialog itself.
     * @param {string} name        unique name of the dialog
     * @returns {function} See usage
     * @private
     */
    function _async(name) {
        return function () {
            var message = '';
            if (typeof arguments[0] === 'string') {
                message = arguments[0];
            }
            dialogs[name].loadingIndicator.loading(message);
            var remainingArguments = message
                ? Array.prototype.slice.call(arguments, 1)
                : arguments;
            return AJS.$.when.apply(this, remainingArguments).then(function () {
                dialogs[name].loadingIndicator.done();
                return AJS.$.Deferred().resolve.apply(this, arguments);
            });
        }
    }

    /**
     * Function that determines if the dialog has been open for a minimum amount of
     * time before closing or removing it.
     * @param {string}   name       Name of the dialog
     * @param {number}   minimumShowDuration    In ms, the minimum amount of time this dialog
     *                                          should be displayed
     * @param {function} callback   Hide or remove function to run after the duration
     *                              is reached.
     * @returns {undefined}
     * @private
     */
    function _minimumShowDuration(name, minimumShowDuration, callback) {
        var now = new Date().getTime();
        var showDuration = now - dialogs[name].showTime;
        if (showDuration < minimumShowDuration) {
            setTimeout(function () {
                executeIfFunction(callback);
            }, minimumShowDuration - showDuration);
        } else {
            executeIfFunction(callback);
        }
    }

    /**
     * Processes the given template parameters and finds any functions an executes them before returning the value.
     * Functions allows us to evaluate parameters at runtime while define them at the time the dialog module is created.
     * @param {object} params The options.templateParameters given to this dialog
     * @returns {object} The processed parameters with any functions applied
     * @private
     */
    function _processParameters(params) {
        var parameters = {};
        for (var key in params) {
            if (params.hasOwnProperty(key)) {
                parameters[key] = _processParameter(params[key]);
            }
        }
        return parameters;
    }

    /**
     * Processes an individual parameter to see if it needs to be executed if it is a function
     * @param {function|*} param Either a function or a plain value
     * @returns {*} the result of processing the parameter
     * @private
     */
    function _processParameter(param) {
        if (typeof param === 'function') {
            return param();
        }

        return param;
    }

    /**
     * Determine what selector to use for the dialog. If a selector option is passed, it will use that. Otherwise,
     * it will use the name of the dialog as the id of the $element
     * @param {string} name        unique name of the dialog
     * @param {object} options     options trickling down from the get method
     * @returns {string} The css selector for the dialog
     * @private
     */
    function _selector(name, options) {
        return options.selector || '#' + name;
    }

    /**
     * Ensures the element exists in the DOM. Creates it if it doesn't exist. Updates the dialog with the
     * element to make sure it is available, and runs init function.
     * @param {string} name        unique name of the dialog
     * @param {object} options     options trickling down from get method
     * @returns {undefined}
     * @private
     */
    function _ensureElement(name, options) {
        if (!dialogs[name].$element) {
            var parameters = _processParameters(options.templateParameters || {});
            var $jQueryDialog = _ensureDialog(_selector(name, options), options.templateName, parameters);
            var loadingIndicator = new LoadingIndicator($jQueryDialog, SPINNER_SELECTOR, HINT_SELECTOR);

            dialogs[name].$element = $jQueryDialog;
            dialogs[name].loadingIndicator = loadingIndicator;

            // Allow the dialog to return a reference to it before executing the init function
            // This makes it so we don't have to pass around a reference to the dialog in the init function
            setTimeout(function () {
                executeIfFunction(options.onInit, $jQueryDialog);
            }, 0);
        }
    }

    /**
     * Ensures that a dialog exists. Otherwise creates the dialog with the specified options
     * @param {string} dialogSelector The jQuery Selector for the dialog in question
     * @param {string} templateName Soy Template Name within the Confluence.Templates.PageHierarchy namespace
     * @param {object} parameters Parameters to pass to the Soy Template
     * @returns {jQuery} the $jQueryDialog reference to this dialog
     */
    function _ensureDialog(dialogSelector, templateName, parameters) {
        var $jQueryDialog = AJS.$(dialogSelector);
        if ($jQueryDialog.length) {
            return $jQueryDialog;
        }
        AJS.$('body').append(Confluence.Templates.PageHierarchy[templateName](parameters));
        $jQueryDialog = AJS.$(dialogSelector);
        return $jQueryDialog;
    }

    /**
     * Returns the dialog object for the given name and ensures that the dialog $element exists in the DOM
     * @param {string} name        unique name of the dialog
     * @param {object} options     options for this dialog
     *      Options:
     *          selector: a specific CSS selector to use instead of the name
     *          templateName: The Soy Template name to use to render the dialog
     *          templateParameters: Parameters to pass to the Soy Template
     *          onInit: init function to run when the dialog is created in the dom.
     *                  It receives the jQuery element as its only parameter
     *          onShow: function to run when the dialog is shown
     *                  It receives the jQuery element as its only parameter
     *          onHide: function to run when the dialog is hidden
     *                  It receives the jQuery element as its only parameter
     *          onRemove: function to run when the dialog is removed
     *                  It receives the jQuery element as its only parameter
     *          showDelay: in ms, the time to wait before actually showing the dialog.
     *                  The dialog onShow function will still run when show is triggered
     *                  on the dialog, but it won't be visible until the delay has passed.
     *                  If the dialog is hidden before the delay time is reached, it
     *                  won't be shown at all. This allows to skip certain dialogs if
     *                  they appear and disappear quickly, without skipping their functionality.
     *                  It creates a nicer visual experience.
     *          minimumShowDuration: in ms, the minimum amount of time that dialog should be
     *                  shown before it is hidden again. If a hide or remove is triggered on
     *                  the dialog before it has reached its minimumShowDuration, the hide or
     *                  remove will be delayed.
     *
     *          Note that you do not need to use the jQuery element parameter in your
     *              init/show/hide/destroy functions, but it is provided as a convenience.
     *              See usage.
     * @returns {object} The dialog
     */
    function get(name, options) {
        if (dialogs[name]) {
            return dialogs[name];
        }

        var dialog = {
            selector: _selector(name, options),
            show: _show(name, options),
            hide: _hide(name, options),
            remove: _remove(name, options),
            async: _async(name)
        };

        dialogs[name] = dialog;

        return dialog;
    }

    return {
        get: get
    }
});