define('confluence/page-hierarchy/dialog/delete-dialog', [
    'ajs',
    'confluence/legacy',
    'confluence/page-hierarchy/fetch/fetch-children-with-incoming-links',
    'confluence/page-hierarchy/fetch/fetch-content-restrictions',
    'confluence/page-hierarchy/fetch/fetch-incoming-link-count',
    'confluence/page-hierarchy/fetch/fetch-space-permissions',
    'confluence/page-hierarchy/fetch/fetch-subtree-count',
    'confluence/page-hierarchy/service/data-service',
    'confluence/page-hierarchy/service/delete-service',
    'confluence/page-hierarchy/service/dialog-service',
    'confluence/page-hierarchy/state/delete-state',
    'confluence/page-hierarchy/util/analytics-event',
    'confluence/page-hierarchy/util/validation-error'
], function (AJS,
             Confluence,
             fetchChildrenWithIncomingLinks,
             fetchRestrictions,
             fetchIncomingLinkCount,
             fetchSpacePermissions,
             fetchSubtreeCount,
             DataService,
             DeleteService,
             DialogService,
             state,
             analyticsEvent,
             validationError) {
    var NAME = 'delete-page-dialog';
    var DELETE_DIALOG_NEXT_SELECTOR = '#delete-dialog-next';
    var DELETE_DIALOG_CLOSE_SELECTOR = '#delete-dialog-close';
    var DELETE_HIERARCHY_CHECKBOX_SELECTOR = '#delete-hierarchy';
    var VARIABLE_DESCRIPTION_SELECTOR = '#variable-description';
    var INCOMING_LINKS_SELECTOR = '#incoming-links';
    var RESTRICTED_PAGE_SELECTOR = '#restricted-page';
    var CHILDREN_WARNING_SELECTOR = '#children-warning';
    var TEMPLATE_NAME = 'deleteDialog';
    var $ = AJS.$;

    var dialog = DialogService.get(NAME, {
        templateName: TEMPLATE_NAME,
        templateParameters: {
            pageTitle: AJS.Meta.get('page-title')
        },
        onShow: _show,
        onInit: _init
    });

    /**
     * Do initial setup actions when the dialog is first loaded. Run once.
     * @returns {undefined}
     * @private
     */
    function _init() {
        _prefetchData();
        _bindEvents();
        _bindButtons();
    }


    /**
     * Main method that runs when this dialog is launched.
     * @returns {undefined}
     * @private
     */
    function _show() {
        _bindValues();
        _determineDescription();
    }

    /**
     * Setup the data prefetchers, so that we can start loading information for the user even
     * before they need it.
     * @returns {undefined}
     * @private
     */
    function _prefetchData() {
        var pageId = AJS.Meta.get('page-id');
        var spaceKey = AJS.Meta.get('space-key');
        DataService.get(DataService.SUBTREE_COUNT_TOTAL_ONLY_DATA_KEY, fetchSubtreeCount(pageId, true));
        DataService.get(DataService.INCOMING_LINKS_DATA_KEY, fetchIncomingLinkCount(pageId));
        DataService.get(DataService.SPACE_PERMISSIONS_DATA_KEY, fetchSpacePermissions(spaceKey));
        DataService.get(DataService.ROOT_INCOMING_LINKS_COUNT_DATA_KEY, fetchChildrenWithIncomingLinks(pageId, pageId));
        DataService.get(DataService.RESTRICTIONS_DATA_KEY, fetchRestrictions(pageId).fail(function(errorResponse){
            validationError.fail(errorResponse, analyticsEvent.DELETE_HIERARCHY_ERROR_EVENT, dialog, state.ERROR_MESSAGES);
        }));
        DataService.get(DataService.SUBTREE_COUNT_DATA_KEY, fetchSubtreeCount(pageId).done(function(subtreeCount) {
            state.setTargetIds(subtreeCount.targetIds);
        }));
    }

    /**
     * Determines the description of the dialog, including any warnings and errors that are present
     * @returns {undefined}
     * @private
     */
    function _determineDescription() {
        var $variableDescription = dialog.$element.find(VARIABLE_DESCRIPTION_SELECTOR);

        dialog.async(
            DataService.get(DataService.SUBTREE_COUNT_TOTAL_ONLY_DATA_KEY),
            DataService.get(DataService.INCOMING_LINKS_DATA_KEY))
            .done(
                function (subtreeCount, numIncomingLinks) {
                    // Subtree count includes the parent page
                    var numDescendants = subtreeCount.totalCount - 1;

                    // No problems :)
                    if (!numIncomingLinks && !numDescendants) {
                        $variableDescription.html(' ' + AJS.I18n.getText('delete.page.hierarchy.description.single.conflict.none'));
                    }
                    _updateChildrenDescription();
                    _updateIncomingLinksWarning(numIncomingLinks);
                    _updateRestrictionsWarning();
                });
    }

    /**
     * Update the warning for if this page is restricted or not and if the children will become visible
     * if it is deleted.
     * @returns {undefined}
     * @private
     */
    function _updateRestrictionsWarning() {
        if (state.getOptions().deleteHierarchy) {
            dialog.async(
                _pageRestricted(),
                DataService.get(DataService.SUBTREE_COUNT_DATA_KEY)
            ).done(function (pageRestricted, subtreeCount) {
                var numDescendants = subtreeCount.totalCount - 1;
                var numRestrictedDescendants = subtreeCount.restrictedCount;
                if (!pageRestricted || numDescendants <= 0) {
                    return;
                }
                _createRestrictionsWarning(numRestrictedDescendants);
            });
        } else {
            dialog.async(
                _pageRestricted(),
                DataService.get(DataService.SUBTREE_COUNT_TOTAL_ONLY_DATA_KEY)
            ).done(function (pageRestricted, subtreeCount) {
                var numDescendants = subtreeCount.totalCount - 1;
                if (!pageRestricted || numDescendants <= 0) {
                    return;
                }
                _createRestrictionsWarning(numDescendants);
            });
        }
    }

    /**
     * Creates the restrictions warning.
     * @param {number} numChildrenLeftBehind The number of children that will be left behind
     * @returns {undefined}
     * @private
     */
    function _createRestrictionsWarning(numChildrenLeftBehind) {
        var $restrictedPage = dialog.$element.find(RESTRICTED_PAGE_SELECTOR);
        $restrictedPage.html(Confluence.Templates.PageHierarchy.restrictedPage({
            numChildrenLeftBehind: numChildrenLeftBehind
        }));
    }

    /**
     * Displays a warning if there are incoming links
     * @param {number} numIncomingLinks The number of incoming links to the parent page
     * @returns {undefined}
     * @private
     */
    function _updateIncomingLinksWarning(numIncomingLinks) {
        if (!numIncomingLinks) {
            return;
        }
        var $incomingLinks = dialog.$element.find(INCOMING_LINKS_SELECTOR);
        $incomingLinks.html(Confluence.Templates.PageHierarchy.incomingLinks({
            numIncomingLinks: numIncomingLinks,
            linkUrl: AJS.contextPath() + '/pages/viewinfo.action?pageId=' + state.getOptions().targetPageId
        }));
    }

    /**
     * Updates the description of how many children will be deleted or left behind based on the user's
     * selection to also delete children or not
     * @returns {undefined}
     * @private
     */

    function _updateChildrenDescription() {
        var $variableDescription = dialog.$element.find(VARIABLE_DESCRIPTION_SELECTOR);
        var numDescendants;
        var numRestrictedDescendants;
        if (state.getOptions().deleteHierarchy) {
            DataService.get(DataService.SUBTREE_COUNT_DATA_KEY).done(function (subtreeCount) {
                // total subtree count includes the parent page;
                numDescendants = subtreeCount.totalCount - 1;
                numRestrictedDescendants = subtreeCount.restrictedCount;

                if (numDescendants) {
                    var numDeleted = numDescendants - numRestrictedDescendants;
                    if (numDeleted === 1) {
                        $variableDescription.html(' ' + AJS.I18n.getText('delete.page.hierarchy.description.single.conflict.orphans.ok.single', numDeleted));
                    } else {
                        $variableDescription.html(' ' + AJS.I18n.getText('delete.page.hierarchy.description.single.conflict.orphans.ok', numDeleted));
                    }
                    if (numRestrictedDescendants > 0) {
                        if (numRestrictedDescendants === 1) {
                            $variableDescription.append(' ' + AJS.I18n.getText('delete.page.hierarchy.description.single.conflict.orphans.single', numRestrictedDescendants));
                        } else {
                            $variableDescription.append(' ' + AJS.I18n.getText('delete.page.hierarchy.description.single.conflict.orphans', numRestrictedDescendants));
                        }
                    }
                }
            });
        } else {
            DataService.get(DataService.SUBTREE_COUNT_TOTAL_ONLY_DATA_KEY).done(function (subtreeCount) {
                // total subtree count includes the parent page;
                numDescendants = subtreeCount.totalCount - 1;
                if (numDescendants === 1) {
                    $variableDescription.html(' ' + AJS.I18n.getText('delete.page.hierarchy.description.single.conflict.orphans.single', numDescendants));
                } else if (numDescendants > 0) {
                    $variableDescription.html(' ' + AJS.I18n.getText('delete.page.hierarchy.description.single.conflict.orphans', numDescendants));
                }
            });
        }
    }

    /**
     * Checks to see if the target page is restricted. It is restricted only if it has view restrictions.
     * Editing restrictions are ignored.
     * @returns {jQuery.Deferred} Promise object with value either true or false
     * @private
     */
    function _pageRestricted() {
        return DataService.get(DataService.RESTRICTIONS_DATA_KEY).pipe(function (data) {
            // Only consider read restrictions since this is what makes pages visible or not.
            var isRestricted = data.read.restrictions.user.size
                || data.read.restrictions.group.size;
            return $.Deferred().resolve(isRestricted);
        });
    }

    /**
     * Determines if the user has legitimate permissions to delete the hierarchy.
     * If a user only has delete own permission, then they are not allowed to delete hierarchies,
     * only their created pages one at a time.
     * @returns {jQuery.Deferred} promise with result either true or false
     * @private
     */
    function _isPermittedToDeleteHierarchy() {
        var deferred = $.Deferred();

        DataService.get(DataService.SPACE_PERMISSIONS_DATA_KEY).done(function (permissions) {
            var hasPermission =
                permissions.systemAdmin
                || permissions.spaceAdmin
                || permissions.deletePages;
            deferred.resolve(hasPermission);
        });

        return deferred.promise();
    }

    /**
     * Bind events to the control buttons for the dialog
     * @returns {undefined}
     * @private
     */
    function _bindButtons() {
        var $deleteDialogNext = dialog.$element.find(DELETE_DIALOG_NEXT_SELECTOR);
        var $deleteDialogClose = dialog.$element.find(DELETE_DIALOG_CLOSE_SELECTOR);

        // Next button binding
        $deleteDialogNext.click(function () {
            if (state.getOptions().deleteHierarchy) {
                AJS.trigger(state.DELETE_HIERARCHY_EVENT);
            } else {
                new DeleteService({
                    loadingIndicator: dialog.loadingIndicator,
                    beforeSubmit: function() {
                        validationError.removeError(dialog);
                    },
                    dialog: dialog
                }).submit();
            }
        });

        // Close button binding
        $deleteDialogClose.click(function (e) {
            e.preventDefault();
            AJS.trigger(state.DELETE_HIERARCHY_CANCEL_EVENT);
        });
    }

    /**
     * Binds any existing values to fields
     * @returns {undefined}
     * @private
     */
    function _bindValues() {
        _bindFields(true);
    }

    /**
     * Binds any events to sync field data. Should be called once.
     * @returns {undefined}
     * @private
     */
    function _bindEvents() {
        _bindFields(false);
    }

    /**
     * Either writes values or binds events to fields
     * @param {boolean} writeValues If true, this method writes values for the fields,
     *                              otherwise it binds its events.
     * @returns {undefined}
     * @private
     */
    function _bindFields(writeValues) {
        var $deleteHierarchy = $(DELETE_HIERARCHY_CHECKBOX_SELECTOR);

        if (writeValues) {
            // trigger change event to update state accordingly
            $deleteHierarchy.change();

            _checkLimit().done(_checkPermissions);
        } else {
            $deleteHierarchy.change(function () {
                state.setDeleteHierarchy($deleteHierarchy.prop('checked'));
                _setNextButtonText();
                _updateChildrenDescription();
                _showOrHideRestrictionWarning();
                _showOrHideIncomingLinksWarning();
            });
        }
    }

    /**
     * Check permissions of the entire hierarchy so the user can see how many pages will be
     * deleted or not
     * @param {boolean} underLimit  Whether we are under the maximum limit
     * @return {undefined}
     * @private
     */
    function _checkPermissions(underLimit) {
        if (!underLimit) {
            return;
        }
        var $childrenWarning = $(CHILDREN_WARNING_SELECTOR);
        $childrenWarning.html(_createWarningIcon(AJS.I18n.getText('delete.page.hierarchy.tooltip.checking.permissions')));
        dialog.async(
            AJS.I18n.getText('delete.page.hierarchy.hint.checking.permissions'),
            _isPermittedToDeleteHierarchy(),
            DataService.get(DataService.SUBTREE_COUNT_DATA_KEY))
            .done(
                function (isPermitted, subtreeCount) {
                    // Total subtree count includes parent page
                    var numDescendants = subtreeCount.totalCount - 1;
                    var numRestrictedDescendants = subtreeCount.restrictedCount;
                    var canDeleteHierarchy = numDescendants > numRestrictedDescendants && isPermitted;
                    _enableOrDisableDeleteHierarchy(canDeleteHierarchy, numDescendants,
                        AJS.I18n.getText('delete.page.hierarchy.also.delete.children.error'));
                    _setNextButtonText();
                }).fail(function(errorResponse) {
                    validationError.fail(errorResponse, analyticsEvent.DELETE_HIERARCHY_ERROR_EVENT, dialog, state.ERROR_MESSAGES);
                });
    }

    /**
     * Check the page limit to see if the user is trying to delete more pages than the maximum.
     * It also disables the delete children checkbox if we are over the limit
     * @returns {jQuery.Deferred} with true if it's okay, or false if we are over the limit
     * @private
     */
    function _checkLimit() {
        var deferred = AJS.$.Deferred();

        dialog.async(DataService.get(DataService.SUBTREE_COUNT_TOTAL_ONLY_DATA_KEY)).done(function (subtreeCount) {
            if (subtreeCount.totalCount > subtreeCount.maximum) {
                _enableOrDisableDeleteHierarchy(false, subtreeCount.totalCount,
                    AJS.I18n.getText('delete.page.hierarchy.page.limit', subtreeCount.maximum));

                // Cannot delete more pages than the limit
                deferred.resolve(false);
            } else {
                deferred.resolve(true);
            }
        });

        return deferred.promise();
    }

    /**
     * Enable or disable the checkbox and text and provide the appropriate message to the user
     * @param {boolean} canDelete True if the user can delete and the checkbox should be enabled,
     *                              false otherwise
     * @param {number} numDescendants The number of descendant pages
     * @param {string} message  The message to display with the warning icon
     * @returns {undefined}
     * @private
     */
    function _enableOrDisableDeleteHierarchy(canDelete, numDescendants, message) {
        var $deleteHierarchy = $(DELETE_HIERARCHY_CHECKBOX_SELECTOR);
        var $childrenWarning = $(CHILDREN_WARNING_SELECTOR);
        var deleteHierarchy = state.getOptions().deleteHierarchy && canDelete;
        var $fieldGroup = $deleteHierarchy.closest('.custom-field-group:first');

        state.setDeleteHierarchy(deleteHierarchy);

        $deleteHierarchy.prop('checked', state.getOptions().deleteHierarchy);
        $deleteHierarchy.prop('disabled', !canDelete);

        if (!canDelete) {
            $fieldGroup.addClass('disabled');
            if (numDescendants === 0) {
                $fieldGroup.hide();
                $childrenWarning.hide();
            } else {
                $childrenWarning.html(_createWarningIcon(message));
            }
        } else {
            $fieldGroup.removeClass('disabled');
            $childrenWarning.empty();
        }
    }

    /**
     * Creates a warning icon for deleting children
     * @param {string} message Message for the tooltip
     * @returns {jQuery} the jQuery warning icon
     * @private
     */
    function _createWarningIcon(message) {
        var $childrenWarning = $('<span class="aui-icon aui-icon-small aui-iconfont-warning warning standalone">Warning</span>');
        $childrenWarning.tooltip({
            title: function () {
                return message;
            }
        });
        return $childrenWarning;
    }

    /**
     * Shows or hides the restriction warning based on whether or not the user has selected to delete child pages also
     * @returns {undefined}
     * @private
     */
    function _showOrHideRestrictionWarning() {
        DataService.get(DataService.SUBTREE_COUNT_DATA_KEY).done(function (subtreeCount) {
            var thereAreChildrenButWeDontDeleteThem =
                (subtreeCount.totalCount - 1 > 0 && !state.getOptions().deleteHierarchy);
            var weDeleteChildrenButSomeCannotBeDeleted =
                (subtreeCount.restrictedCount > 0 && state.getOptions().deleteHierarchy);
            var show = thereAreChildrenButWeDontDeleteThem
                || weDeleteChildrenButSomeCannotBeDeleted;
            var $restrictedPage = dialog.$element.find(RESTRICTED_PAGE_SELECTOR);
            $restrictedPage.toggle(show);
        });
    }

    /**
     * Shows or hides the incoming links warning based on whether or not the user has selected to delete child pages also
     * @returns {undefined}
     * @private
     */
    function _showOrHideIncomingLinksWarning() {
        var show = !state.getOptions().deleteHierarchy;
        var $incomingLinks = dialog.$element.find(INCOMING_LINKS_SELECTOR);
        $incomingLinks.toggle(show);
    }

    /**
     * Set the text of the next button
     * @returns {undefined}
     * @private
     */
    function _setNextButtonText() {
        var $deleteDialogNext = dialog.$element.find(DELETE_DIALOG_NEXT_SELECTOR);
        if (state.getOptions().deleteHierarchy) {
            $deleteDialogNext.html(AJS.I18n.getText('copy.page.dialog.next.button'));
        } else {
            $deleteDialogNext.html(AJS.I18n.getText('delete.page.hierarchy.dialog.delete'));
        }
    }

    return dialog;
});