define('confluence/page-hierarchy/util/loading-indicator', [
    'ajs'
], function (AJS) {

    /**
     * Loading indicator utility that displays a spinner and optional message within the specified parameters
     * @param {jQuery} $context Parent DOM element that is the current context of this loadingIndicator.
     * @param {string} spinnerSelector A jQuery selector for a spinner in this context
     * @param {string} messageSelector A jQuery selector for an element where a message might be displayed in this
     *                                  context
     * @constructor
     */
    function LoadingIndicator($context, spinnerSelector, messageSelector) {
        this.$messageBox = messageSelector && $context.find(messageSelector);
        this.$spinner = spinnerSelector && $context.find(spinnerSelector);
        this.count = 0;
    }

    /**
     * Set the loading state of this indicator to 'loading'
     * @param {string} message Optional message to display
     * @returns {undefined}
     */
    LoadingIndicator.prototype.loading = function (message) {
        this.count++;
        this.$spinner && this.$spinner.spin();
        if (message) {
            this.$messageBox && this.$messageBox.html(AJS.escapeEntities(message));
        }
    };

    /**
     * Resets the loading state of this indicator to 'not loading'
     * It takes into account how many times the indicator loading() method
     * has been called, and only resets itself if all the loading events have finished.
     * @param {function} callback If given, will be execute if we are truly finished loading
     * @returns {undefined}
     */
    LoadingIndicator.prototype.done = function (callback) {
        if (--this.count <= 0) {
            this.count = 0;
            this.$messageBox && this.$messageBox.empty();
            this.$spinner && this.$spinner.spinStop();
            if (callback && typeof callback === 'function') {
                callback();
            }
        }
    };

    return LoadingIndicator;
});