/**
 * Sets up the dialog with the options to copy a page with its children
 * including any relevant options.
 */
define('confluence/page-hierarchy/dialog/delete-hierarchy-dialog', [
    'ajs',
    'confluence/legacy',
    'confluence/page-hierarchy/action/delete-page-hierarchy-action',
    'confluence/page-hierarchy/service/data-service',
    'confluence/page-hierarchy/service/delete-service',
    'confluence/page-hierarchy/service/dialog-service',
    'confluence/page-hierarchy/service/page-tree',
    'confluence/page-hierarchy/state/delete-state',
    'confluence/page-hierarchy/util/validation-error'
], function (AJS,
             Confluence,
             deletePageHierarchyAction,
             DataService,
             DeleteService,
             DialogService,
             PageTree,
             state,
             validationError
) {
    var NAME = 'delete-hierarchy-dialog';
    var SUBMIT_BUTTON_SELECTOR = '#delete-hierarchy-submit';
    var BACK_BUTTON_SELECTOR = '#delete-hierarchy-back';
    var CLOSE_BUTTON_SELECTOR = '#delete-hierarchy-close';
    var PREVIEW_CURRENT_PAGE_SELECTOR = '#currentPage';
    var PREVIEW_SELECTOR = '#preview';
    var PREVIEW_CHILDREN_SELECTOR = '#children';
    var DESCRIPTION_SELECTOR = '#description';
    var TEMPLATE_NAME = 'deleteHierarchyDialog';

    var dialog = DialogService.get(NAME, {
        templateName: TEMPLATE_NAME,
        templateParameters: {
            currentPageTitle: AJS.Meta.get('page-title')
        },
        onInit: _init
    });



    /**
     * Init method runs when the dialog is created
     * @returns {undefined}
     * @private
     */
    function _init() {
        _bindButtons();
        _setupPreview();
        _updateDescriptionOfWhatIsBeingDeleted();

        var id = '#' + NAME;

        AJS.dialog2(id).on('hide', function () {
            validationError.removeError(dialog);
        });

        AJS.dialog2(id).on('show', function () {
            _setupPreview();
        });
    }

    /**
     * Sets up the preview with the parent page and all the children.
     * @returns {undefined}
     * @private
     */
    function _setupPreview() {
        var $preview = dialog.$element.find(PREVIEW_SELECTOR);
        var $childrenElement = $preview.find(PREVIEW_CHILDREN_SELECTOR);
        var pageId = AJS.Meta.get('page-id');

        var pageTree = new PageTree({
            loadingIndicator: dialog.loadingIndicator,
            incomingLinks: true,
            restrictEditPages: true,
            rootContentId: state.getOptions().targetPageId
        });
        pageTree.load(pageId, $childrenElement);
        _updateRootIncomingLinksCount();
    }


    /**
     * Update the incoming links information for the root element in our tree
     * @returns {undefined}
     * @private
     */
    function _updateRootIncomingLinksCount() {
        var pageId = state.getOptions().targetPageId;
        var $parentPage = dialog.$element.find(PREVIEW_CURRENT_PAGE_SELECTOR);
        var $parentPageTitle = $parentPage.find('.title').first();
        dialog.async(DataService.get(DataService.ROOT_INCOMING_LINKS_COUNT_DATA_KEY)).done(function (children) {
            var parentLinkCount = children.results[0].metadata.incomingCountForParent;
            var $incomingLinkCountLink = $parentPageTitle.find(".incoming-links").first();
            if ($incomingLinkCountLink.length) {
                $incomingLinkCountLink.replaceWith(Confluence.Templates.PageHierarchy.previewIncomingLinks({
                    pageId: pageId,
                    linkCount: parentLinkCount
                }));
            } else {
                $parentPageTitle.append(Confluence.Templates.PageHierarchy.previewIncomingLinks({
                    pageId: pageId,
                    linkCount: parentLinkCount
                }));
            }
        });
    }

    /**
     * Figures out how many pages are going to be deleted, and left behind, and how many
     * pages have incoming links
     * @returns {undefined}
     * @private
     */
    function _updateDescriptionOfWhatIsBeingDeleted() {
        var $description = dialog.$element.find(DESCRIPTION_SELECTOR);
        var escapedPageTitle = AJS.escapeEntities(AJS.Meta.get('page-title'));

        DataService.get(DataService.SUBTREE_COUNT_DATA_KEY).done(function (subtreeCount) {
            // total count includes the parent page
            var numDescendants = subtreeCount.totalCount - 1;
            var numRestrictedDescendants = subtreeCount.restrictedCount;

            var numToDelete = numDescendants - numRestrictedDescendants;
            var description = '';

            if (numToDelete === 1) {
                description = AJS.I18n.getText('delete.page.hierarchy.description.multiple.page', escapedPageTitle, numToDelete);
            } else {
                description = AJS.I18n.getText('delete.page.hierarchy.description.multiple.pages', escapedPageTitle, numToDelete);
            }
            if (numRestrictedDescendants > 0) {
                if (numRestrictedDescendants === 1) {
                    description += ' ' + AJS.I18n.getText('delete.page.hierarchy.description.single.conflict.orphans.single', numRestrictedDescendants);
                } else {
                    description += ' ' + AJS.I18n.getText('delete.page.hierarchy.description.single.conflict.orphans', numRestrictedDescendants);
                }
            }
            // TODO incoming links
            $description.html(description);
        });
    }

    /**
     * Bind events to the control buttons for this dialog
     * @returns {undefined}
     * @private
     */
    function _bindButtons() {
        var $submitButton = dialog.$element.find(SUBMIT_BUTTON_SELECTOR);
        var $backButton = dialog.$element.find(BACK_BUTTON_SELECTOR);
        var $closeButton = dialog.$element.find(CLOSE_BUTTON_SELECTOR);

        $backButton.click(function () {
            AJS.trigger(state.DELETE_HIERARCHY_BACK_EVENT);
        });

        $submitButton.click(function () {
            new DeleteService({
                loadingIndicator: dialog.loadingIndicator,
                beforeSubmit: function() {
                    validationError.removeError(dialog);
                },
                dialog: dialog
            }).submit();
        });

        $closeButton.click(function (e) {
            e.preventDefault();
            AJS.trigger(state.DELETE_HIERARCHY_CANCEL_EVENT);
        });
    }

    return dialog;
});