package com.atlassian.confluence.plugins.mentions;

import com.atlassian.confluence.content.render.xhtml.ResettableXmlEventReader;
import com.atlassian.confluence.content.render.xhtml.XMLEventFactoryProvider;
import com.atlassian.confluence.content.render.xhtml.XmlEventReaderFactory;
import com.atlassian.confluence.content.render.xhtml.XmlOutputFactory;
import com.atlassian.confluence.core.BodyContent;
import com.atlassian.confluence.core.BodyType;
import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.user.ConfluenceUser;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventFactory;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Attribute;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.atlassian.confluence.content.render.xhtml.StaxUtils.closeQuietly;
import static com.atlassian.confluence.content.render.xhtml.XhtmlConstants.RESOURCE_IDENTIFIER_NAMESPACE_PREFIX;
import static com.atlassian.confluence.content.render.xhtml.XhtmlConstants.RESOURCE_IDENTIFIER_NAMESPACE_URI;
import static com.atlassian.confluence.content.render.xhtml.XhtmlConstants.XHTML_NAMESPACE_URI;
import static com.atlassian.confluence.content.render.xhtml.storage.inlinetask.StorageInlineTaskConstants.TASK_ELEMENT;
import static com.atlassian.confluence.content.render.xhtml.storage.inlinetask.StorageInlineTaskConstants.TASK_LIST_ELEMENT;
import static com.atlassian.confluence.content.render.xhtml.storage.resource.identifiers.StorageResourceIdentifierConstants.USERKEY_ATTRIBUTE_NAME;
import static com.atlassian.confluence.content.render.xhtml.storage.resource.identifiers.StorageResourceIdentifierConstants.USER_RESOURCE_QNAME;
import static com.google.common.collect.Sets.newHashSet;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * A StAX based excerptor which uses a recursive approach to generating excerpts that mention a specified user.
 */
public class StaxMentionsExcerptor implements MentionsExcerptor {
    private static final QName USERKEY_ATTRIBUTE_QNAME = new QName(
            RESOURCE_IDENTIFIER_NAMESPACE_URI,
            USERKEY_ATTRIBUTE_NAME,
            RESOURCE_IDENTIFIER_NAMESPACE_PREFIX);

    private static final QName PARAGRAPH_QNAME = new QName(XHTML_NAMESPACE_URI, "p");
    private static final QName LI_QNAME = new QName(XHTML_NAMESPACE_URI, "li");
    private static final QName TR_QNAME = new QName(XHTML_NAMESPACE_URI, "tr");
    private static final Set<QName> UNIT_ELEMENTS = newHashSet(
            PARAGRAPH_QNAME,
            LI_QNAME,
            TR_QNAME,
            TASK_ELEMENT);

    private static final Set<QName> LIST_ELEMENTS = newHashSet(
            new QName(XHTML_NAMESPACE_URI, "ol"),
            new QName(XHTML_NAMESPACE_URI, "ul"),
            TASK_LIST_ELEMENT);

    private static final Map<QName, String> DIVIDERS = new HashMap<>();

    static {
        DIVIDERS.put(PARAGRAPH_QNAME, "<p style=\"text-align: left\">&middot;&middot;&middot;</p>");
    }

    private final XmlEventReaderFactory xmlEventReaderFactory;
    private final XmlOutputFactory xmlFragmentOutputFactory;
    private final XMLEventFactory xmlEventFactory;

    public StaxMentionsExcerptor(XmlEventReaderFactory xmlEventReaderFactory, @Qualifier("xmlFragmentOutputFactory") XmlOutputFactory xmlFragmentOutputFactory, XMLEventFactoryProvider xmlEventFactoryProvider) {
        this.xmlEventReaderFactory = xmlEventReaderFactory;
        this.xmlEventFactory = xmlEventFactoryProvider.getXmlEventFactory();
        this.xmlFragmentOutputFactory = xmlFragmentOutputFactory;
    }

    @Override
    public String getExcerpt(ContentEntityObject content, ConfluenceUser mentionedUser) {
        if (mentionedUser == null || isBlank(mentionedUser.getName()) || content == null) {
            return "";
        }

        BodyContent bodyContent = content.getBodyContent();

        if (bodyContent.getBodyType() != BodyType.XHTML) {
            return "";
        }

        XMLEventReader reader = null;
        List<? extends CharSequence> excerpts;
        try {
            reader = xmlEventReaderFactory.createStorageXmlEventReader(
                    new StringReader("<div>" + bodyContent.getBody() + "</div>")
            );
            excerpts = getExcerpts(reader, mentionedUser);
        } catch (XMLStreamException exception) {
            throw new RuntimeException("Error occurred while reading stream", exception);
        } finally {
            closeQuietly(reader);
        }

        StringBuilder result = new StringBuilder();
        excerpts.forEach(result::append);

        return result.toString();
    }

    /**
     * Extract an excerpt for the specified fragment
     *
     * @param xmlFragment a reader over the xml fragment. This method will consume all the events in this fragment when it returns. Pointer of reader should be just before start element.
     * @param user        the user to look for
     * @return an excerpt for the specified fragment
     * @throws XMLStreamException
     */
    private List<? extends CharSequence> getExcerpts(XMLEventReader xmlFragment, ConfluenceUser user) throws XMLStreamException {
        if (xmlFragment == null || xmlFragment.peek() == null) {
            return emptyList();
        }

        if (!xmlFragment.peek().isStartElement()) {
            throw new IllegalArgumentException("xmlFragmentReader should serve start element of the fragment first.");
        }

        try {
            StartElement xmlFragmentStart = xmlFragment.peek().asStartElement();

            if (UNIT_ELEMENTS.contains(xmlFragmentStart.getName())) {
                xmlFragment = new ResettableXmlEventReader(xmlFragment);

                if (fragmentContainsUser(xmlFragment, user)) {
                    ((ResettableXmlEventReader) xmlFragment).reset();
                    return singletonList(toStringBuilder(xmlFragment));
                } else {
                    String divider = DIVIDERS.get(xmlFragmentStart.getName());
                    return singletonList(divider == null ? "" : divider);
                }
            } else if (LIST_ELEMENTS.contains(xmlFragmentStart.getName())) {
                LinkedList<CharSequence> excerpts = new LinkedList<>();
                XMLEventReader xmlFragmentBody = xmlEventReaderFactory.createXmlFragmentBodyEventReader(xmlFragment);

                while (xmlFragmentBody.hasNext()) {
                    XMLEvent xmlEvent = xmlFragmentBody.peek();

                    if (xmlEvent.isStartElement() && UNIT_ELEMENTS.contains(xmlEvent.asStartElement().getName())) {
                        XMLEventReader unitFragment = xmlEventReaderFactory.createXmlFragmentEventReader(xmlFragmentBody);

                        getExcerpts(unitFragment, user).stream()
                                .filter(excerpt -> excerpt != excerpts.peekLast() && !"".equals(excerpt))
                                .forEach(excerpts::add);
                    } else {
                        xmlFragmentBody.nextEvent(); // consume all elements in this xmlFragmentBody
                    }
                }

                if (excerpts.size() > 0) {
                    StringWriter listFragmentBuffer = new StringWriter();
                    XMLEventWriter listFragment = xmlFragmentOutputFactory.createXMLEventWriter(listFragmentBuffer);

                    try {
                        listFragment.add(xmlFragmentStart);
                        listFragment.add(xmlEventFactory.createCharacters("")); // to cause ">" to be appended
                        listFragment.flush();

                        excerpts.add(0, new StringBuilder(listFragmentBuffer.getBuffer()));

                        int position = listFragmentBuffer.getBuffer().length();
                        listFragment.add(xmlFragment.nextEvent()); // consume xmlFragmentEnd
                        listFragment.flush();
                        excerpts.add(new StringBuilder(listFragmentBuffer.getBuffer().subSequence(position, listFragmentBuffer.getBuffer().length())));
                    } finally {
                        closeQuietly(listFragment);
                    }

                    return excerpts;
                }
            } else if ("table".equals(xmlFragmentStart.getName().getLocalPart())) {
                LinkedList<CharSequence> excerpts = new LinkedList<>();

                List<XMLEvent> tableEvents = new LinkedList<>();

                tableEvents.add(xmlFragment.nextEvent()); // consume start table event

                while (xmlFragment.hasNext()) {
                    XMLEvent xmlEvent = xmlFragment.peek();

                    if (xmlEvent.isStartElement() && "tr".equals(xmlEvent.asStartElement().getName().getLocalPart())) {
                        XMLEventReader trFragment = xmlEventReaderFactory.createXmlFragmentEventReader(xmlFragment);
                        getExcerpts(trFragment, user).stream()
                                .filter(excerpt -> excerpt != excerpts.peekLast() && !"".equals(excerpt))
                                .forEach(excerpts::add);
                    } else {
                        tableEvents.add(xmlFragment.nextEvent());
                    }
                }

                /*
                 * The following block could probably be written better. Please feel free to if you have any better ideas.
                 */
                if (excerpts.size() > 0) {
                    StringWriter tableFragmentBuffer = new StringWriter();
                    XMLEventWriter tableFragment = xmlFragmentOutputFactory.createXMLEventWriter(tableFragmentBuffer);

                    try {
                        for (XMLEvent tableEvent : tableEvents) {
                            if (tableEvent.isStartElement() && "tbody".equals(tableEvent.asStartElement().getName().getLocalPart())) {
                                tableFragment.add(tableEvent);
                                tableFragment.add(xmlEventFactory.createCharacters(""));
                                tableFragment.flush(); // forces <tbody></tbody> to be written instead of <tbody />
                            } else {
                                tableFragment.add(tableEvent);
                            }
                        }
                    } finally {
                        closeQuietly(tableFragment);
                    }

                    int indexAfterTbody = tableFragmentBuffer.getBuffer().indexOf("<tbody>") + "<tbody>".length();
                    excerpts.add(0, new StringBuilder(tableFragmentBuffer.getBuffer().subSequence(0, indexAfterTbody)));
                    excerpts.add(new StringBuilder(tableFragmentBuffer.getBuffer().subSequence(indexAfterTbody, tableFragmentBuffer.getBuffer().length())));

                    return excerpts;
                }
            } else {
                xmlFragment.nextEvent(); // consume start element

                LinkedList<CharSequence> excerpts = new LinkedList<>();

                while (xmlFragment.hasNext()) {
                    XMLEvent xmlFragmentEvent = xmlFragment.peek();

                    if (xmlFragmentEvent.isStartElement()) {
                        getExcerpts(xmlEventReaderFactory.createXmlFragmentEventReader(xmlFragment), user).stream()
                                .filter(excerpt -> excerpt != excerpts.peekLast())
                                .forEach(excerpts::add);
                    } else {
                        xmlFragment.nextEvent();
                    }
                }

                return excerpts;
            }

            return emptyList();
        } finally {
            // always ensure we read all events in the fragment
            closeQuietly(xmlFragment);
        }
    }

    private boolean fragmentContainsUser(XMLEventReader xmlFragmentReader, ConfluenceUser user) throws XMLStreamException {
        while (xmlFragmentReader.hasNext()) {
            final XMLEvent xmlEvent = xmlFragmentReader.nextEvent();

            if (xmlEvent.isStartElement()) {
                final StartElement startElement = xmlEvent.asStartElement();

                if (startElement.getName().equals(USER_RESOURCE_QNAME)) {
                    final Attribute userKeyAttribute = startElement.getAttributeByName(USERKEY_ATTRIBUTE_QNAME);
                    if (userKeyAttribute != null && user.getKey() != null) {
                        if (user.getKey().getStringValue().equals(userKeyAttribute.getValue())) {
                            return true;
                        }
                    }
                }
            }
        }

        return false;
    }

    private CharSequence toStringBuilder(XMLEventReader xmlFragmentReader) {
        XMLEventWriter xmlEventWriter = null;
        StringWriter result = new StringWriter();
        try {
            xmlEventWriter = xmlFragmentOutputFactory.createXMLEventWriter(result);
            xmlEventWriter.add(xmlFragmentReader);
        } catch (XMLStreamException e) {
            throw new RuntimeException(e);
        } finally {
            closeQuietly(xmlEventWriter);
        }

        return result.toString();
    }
}
