define('confluence/cjc/error/error-message-view', [
    'jquery',
    'underscore',
    'ajs',
    'backbone',
    'confluence/cjc/amd/template',
    'confluence/cjc/error/error-type'
],
function(
    $,
    _,
    AJS,
    Backbone,
    template,
    ErrorType
) {
    'use strict';

    var ErrorMessageView = Backbone.View.extend({

        initialize: function() {
            var ErrorTypes = Backbone.Collection.extend({
                model: ErrorType
            });

            this.collection = new ErrorTypes();

            this.listenTo(this.collection, 'reset', this._handleClearAll);
            this.listenTo(this.collection, 'add', this._handleAddError);
            this.listenTo(this.collection, 'remove', this._handleRemoveError);

            this.on('change', function() {
                var $errors = this.$el.find('li');

                // hide parent div if all children .message div are hidden
                this.$el.toggleClass('hidden', $errors.length === 0);
                this.$el.toggleClass('single-error', $errors.length === 1);

            }, this);

            this.render();
            return this;
        },

        render: function() {
            // render AUI message HTML container for inner messages
            var $message = AJS.messages.warning({
                body: '<ul></ul>',
                closeable: false
            });

            this.$el.append($message);
        },

         /**
         * Show a warning message
         * @param msg body of message to show
         * @param typeName Id of error message
         */
        addMessage: function(typeName, msg) {
            if (typeof typeName === 'string') {
                var newError = new ErrorType({
                    type: typeName,
                    errorMessageContent: msg || ''
                });
                this.collection.add(newError);
            } else if (typeName && typeName.length) {
                this.collection.add(typeName);
            }

            this.trigger('change');
        },

        /**
         * Hide a warning message
         */
        removeMessage: function(typeName) {
            var model = this.collection.where({
                type: typeName
            });
            this.collection.remove(model);

            this.trigger('change');
        },

        _handleClearAll: function() {
            this.$el.find('li').remove();
            this.trigger('change');
        },

        _handleAddError: function(model) {
            var markup = template.errorMessage(model.toJSON());
            this.$el.find('ul').append(markup);
        },

        _handleRemoveError: function(model) {
            this.$el.find('ul').find('li.' + model.get('type')).remove();
        },

        hasError: function() {
            return this.collection.any(function(model) {
                if (model.get('isWarning') === false) {
                    return true;
                }
            });
        },

        hasCritical: function() {
            return this.collection.any(function(model) {
                if (model.get('isCritical')) {
                    return true;
                }
            });
        },

        /**
         * Remove all error messages, except error message has `isPersistent=true`
         */
        reset: function() {
            var self = this;

            this.collection.each(function(errorModel) {
                if (!errorModel.get('isPersistent')) {
                    self.collection.remove(errorModel);
                }
            });

            this.trigger('change');
        }
    });

    return ErrorMessageView;
});
