define('confluence/cjc/create-issue-form/extra-required-fields-view', [
    'ajs',
    'jquery',
    'underscore',
    'backbone',
    'confluence/cjc/analytics',
    'confluence/cjc/error/error-type',
    'confluence/cjc/jira-issue-data-helper',
    'confluence/cjc/common/services'
],
function(
    AJS,
    $,
    _,
    Backbone,
    analytics,
    ErrorType,
    jiraIssueDataHelper,
    services
) {
    'use strict';

    /**
     * Rendering extra required fields.
     * This class is migrated from a part of `.../js/jira-form-helper.js`.
     * We want to kill `.../js/jira-form-helper.js` and first step to split it up as many modules as possible.
     */
    var ExtraRequiredFieldsView = Backbone.View.extend({
        template: window.Confluence.CreateJiraContent,

        initialize: function(options) {
            // get all necessary input options
            this.formObjectView = options.formObjectView;

            // initialize variables
            this.$formWrapper = this.formObjectView.$el;
            this.$serverSelect = this.$formWrapper.find('#jira-servers');
            this.$projectSelect = this.$formWrapper.find('#jira-projects');
            this.$issueTypeSelect = this.$formWrapper.find('#jira-issue-types');
            this.$summaryField = this.$formWrapper.find('#issue-summary');
            this.$descriptionField = this.$formWrapper.find('#issue-description');

            this.$currentRequiredFields = null;
            this.hasUnsupportedFields = false;
        },

        render: function(otherRequiredFields) {
            services.renderCreateRequiredFields({
                $container: this.$el,
                afterElement: null,
                context: {
                    serverId: this.$serverSelect.val(),
                    projectKey: this._getCurrentProject().key,
                    issueType: this.$issueTypeSelect.val()
                },
                options: {
                    requiredFields: otherRequiredFields
                },
                errorCallback: _.bind(this._showUnsupportedFieldsMessage, this)
            });

            // update current required fields
            this.$currentRequiredFields = this.$('.jira-field').children('input,select,textarea');

            // add some attributes to support validation
            this.$currentRequiredFields.each(function(index, el) {
                var $el = $(el);

                // turn off browser autocomplete for date textbox
                var isDateField = $el.closest('[data-jira-type*=duedate]').length > 0 ||
                        $el.closest('[data-jira-type*=datepicker]').length > 0;

                if (isDateField) {
                    $el.attr('autocomplete', 'off');
                }

                // add `required` attribute so that `.../create-issue-form-submit-validation.js` can do its job well.
                var fieldName = $el.parent().find('label').text();
                if (fieldName) {
                    $el.attr('required', 'required');
                    $el.attr('data-name', fieldName);
                }
            });

            this._fixOldAUIDatePicker();
        },

        _fixOldAUIDatePicker: function() {
            // The block of code bellow to override the behavior of onSelect function.
            // As default, datepicker doesn't fire change event, this override will fix this issue.
            // This override only apply for AUI versions which < 5.4.5
            if (AJS.version <= '5.4.5' ) {
                var $datepickers = this.$('input[data-aui-dp-uuid]');

                $datepickers.each(function(index, element) {
                    var $dp = $(element),
                        uuid = $dp.attr('data-aui-dp-uuid');

                    $dp.on('click', function() {
                        var $dpPopup = $('[data-aui-dp-popup-uuid=' + uuid + ']');
                        var defaultOnSelectHandler = $dpPopup.datepicker('option', 'onSelect');

                        $dpPopup.datepicker('option', 'onSelect', function(dateText, inst) {
                            defaultOnSelectHandler(dateText, inst);
                            $dp.change();
                        });
                        $dpPopup.parents('.aui-inline-dialog').addClass('datepicker-patch');
                    });
                });
            }
        },

        _showUnsupportedFieldsMessage: function(unsupportedFields) {
            this.hasUnsupportedFields = true;

            var markup = this.template.renderUnsupportedFieldsWarningMessage({
                unsupportedFields: _.map(unsupportedFields, function(item) {
                    return item.name;
                }),
                projectLinkUrl: this._getCurrentJiraCreateIssueUrl()
            });

            this.formObjectView.errorMessageView.addMessage(ErrorType.ERROR_FORM_ID, markup);

            analytics.sendAnalyticsForRequiredFields();
            analytics.bindAnalyticsForCreateDirectlyInJira();

            this.$formWrapper.find('#issue-content, #jira-epic-content').addClass('hidden');
            this.formObjectView.refreshDialog();
        },

        _getCurrentProject: function() {
            var serverId = this.$serverSelect.val();
            var projectId = this.$projectSelect.val();
            var projects = jiraIssueDataHelper.getProjects(serverId);

            return _.find(projects, function(project) {
                return project.id === projectId;
            });
        },

        _getCurrentJiraCreateIssueUrl: function() {
            var jiraCreateIssueUrl = AJS.format('{0}/secure/CreateIssueDetails!Init.jspa?pid={1}&issuetype={2}',
                                        this.formObjectView.serverSelectView.getCurrentServer().url,
                                        this._getCurrentProject().id,
                                        this.$issueTypeSelect.val());

            var summaryValue =  this.$summaryField.val();
            if (summaryValue.length) {
                jiraCreateIssueUrl = jiraCreateIssueUrl + '&summary=' + encodeURIComponent(summaryValue);
            }

            if (this.formObjectView.currentDescriptionFieldStatus != window.Confluence.CreateJiraContent.JiraIssue.FieldStatus.HIDDEN) {
                var descriptionValue = this.$descriptionField.val();
                if (descriptionValue.length) {
                    jiraCreateIssueUrl = jiraCreateIssueUrl + '&description=' + encodeURIComponent(descriptionValue);
                }
            }
            return jiraCreateIssueUrl;
        },

        fillRequiredFieldsToObject: function(json) {

            if (this.$currentRequiredFields) {
                this.$currentRequiredFields.each(function(index, requiredField) {
                    var $requiredField = $(requiredField);
                    json.fields[$requiredField.attr('name')] = window.jiraIntegration.fields.getJSON($requiredField);
                });
            }

            return json;
        },

        reset: function() {
            this.$currentRequiredFields = null;
            this.hasUnsupportedFields = false;
            this.$el.empty();
        }

    });

    return ExtraRequiredFieldsView;
});
