define('confluence/cjc/create-issue-form/create-multiples-issues-view', [
    'ajs',
    'underscore',
    'confluence/cjc/create-issue-form/create-issue-form-view-abstract',
    'confluence/cjc/table-content-view',
    'confluence/cjc/error/error-type',
    'confluence/cjc/analytics'
],
function(
    AJS,
    _,
    CreateIssueFormViewAbstract,
    TableContentView,
    ErrorType,
    analytics
) {
    'use strict';

    /**
     * Creating Multiple Issue View
     */
    var CreateMultipleIssuesView = CreateIssueFormViewAbstract.extend({
        template: window.Confluence.CreateJiraContent,

        initialize: function(options) {
            CreateIssueFormViewAbstract.prototype.initialize.apply(this, arguments);

            this.tableContentView = new TableContentView({
                el: options.tableElement,
                mappingFields: this.mappingFields
            });

            this._prepareDataForCreatingFromTable();
        },

        render: function() {
            var self = this;
            CreateIssueFormViewAbstract.prototype.render.apply(this, arguments);

            this.validate();
            this._bindOnSelectChangeEvent();

            this.listenTo(this.formObjectView, 'confluence-jira-content.form-updated-fieldmeta', function(metaFields) {
                self._collectDataFromTableAndValidate();
            });
        },

        /**
         * Scan creating single issue form and create IssueCollection
         * @returns {IssueCollection}
         */
        getIssueCollection: function() {
            var issueCollection = this.tableContentView.extractIssuesFromTable();

            issueCollection.each(function(issueModel) {
                this.extractExtraFields(issueModel);
            }.bind(this));

            return issueCollection;
        },

        _prepareDataForCreatingFromTable: function() {
            var summaryMappingFieldModel = this.mappingFields.getIssueFieldByName('summary');
            var indexColumn = $(this.selectionObject.containingElement).closest('td').index();
            summaryMappingFieldModel.set('index', indexColumn);

            // when select Multiple Cell or Double Click on cell in Chrome (startContainer contains the correct element)
            // =>try to get first element if support (Firefox, chrome)
            if (summaryMappingFieldModel.get('index') === -1 && this.selectionObject.range.startContainer) {
                var containingElement = $(this.selectionObject.range.startContainer);
                indexColumn = containingElement.closest('td').index();
                summaryMappingFieldModel.set('index', indexColumn);
            }

            // if cannot detect the cell selected (multiple cell in IE<=8)
            if (summaryMappingFieldModel.get('index') === -1) {
                summaryMappingFieldModel.set('index', 0);
            }

            // List table header and index
            var listColumnInTable = this.tableContentView.getListColumns();
            var totalColumn = listColumnInTable.length;

            var descMappingFieldModel = this.mappingFields.getIssueFieldByName('description');
            descMappingFieldModel.setNextIndexColumn(summaryMappingFieldModel.get('index'), totalColumn);
        },

        fillPreviewIssuesFromTable: function() {
            if (this.formObjectView.isInTextMode()) {
                return;
            }

            var $issueTablePlaceholder = this.$('#create-from-table');
            $issueTablePlaceholder.empty();
            if (this.$('#jira-projects option:selected').val() !== -1) {

                var issueCollection = this.getIssueCollection();
                var columnIssues = issueCollection.getFieldNameAndContentAsArray();

                if (columnIssues.length === 0) {
                    return;
                }

                var numberOfRemainingIssues = 0;

                // clone array
                var slicedColumnIssues = columnIssues.slice(0);

                if (slicedColumnIssues.length > CreateMultipleIssuesView.MAX_ISSUE_PREVIEW) {
                    slicedColumnIssues = slicedColumnIssues.slice(0, CreateMultipleIssuesView.MAX_ISSUE_PREVIEW);
                    numberOfRemainingIssues = columnIssues.length - CreateMultipleIssuesView.MAX_ISSUE_PREVIEW;
                }

                var markup = this.template.createPreviewIssuesFromTable({
                    issues: slicedColumnIssues,
                    numberOfRemainingIssues: numberOfRemainingIssues
                });

                $issueTablePlaceholder.html(markup);
            }
            var issueTypeSelected = this.$('#jira-issue-types option:selected').val();
            if (issueTypeSelected !== -1) {
                this.formObjectView.changeIssuesTypeIconForPreviewPanel(this.$('#jira-issue-types option:selected').val());
            }

            // refresh dialog to resize panel after remove content in dialog
            this.formObjectView.refreshDialog();
        },

        refresh: function() {
            var $dialogHeader = this.$('#create-issues-dialog-header');
            $dialogHeader.text(AJS.I18n.getText('createjiracontent.dialog.form.header.title.plural'));

            this.fillPreviewIssuesFromTable();

            // fill text for the link to switch to creating single issue form
            this.toggleLinkSwitchSingleMultipleForm(true);

            this.$('#create-from-text').hide();
            this.$('#create-from-table, #create-from-table-field').show();

            // can not detect header of table
            if (!this.formObjectView.listColumnInTable.length) {
                this.formObjectView.errorMessageView.addMessage(ErrorType.ERROR_CAN_NOT_DETECT_TABLE_HEADER);
            }
        },

        _collectDataFromTableAndValidate: function() {
            var valid = true;

            // update index
            this.mappingFields.each(function(mappingFieldModel) {
                var fieldIdSelector = '#issue-' + mappingFieldModel.get('name') + '-index';
                var $select = $(fieldIdSelector);
                var index = window.parseInt($select.val());

                if (!isNaN(index)) {
                    mappingFieldModel.set('index', index);
                }
            });

            var issueCollection = this.getIssueCollection();
            var errors = issueCollection.validate();
            this.renderErrorMessages(errors);

            valid = errors.length === 0;

            return valid;
        },

        validate: function() {
            // handling error message in sorted table
            if (this.tableContentView && this.tableContentView.isInSortedTable()) {
                this.formObjectView.errorMessageView.addMessage(ErrorType.WARNING_SORTED_TABLE_ID);
                analytics.sendAnalyticsForSortedTable();
                return false;
            }

            return true;
        },

        submitFormWhenValid: function() {
            this.formObjectView.errorMessageView.reset();

            this._collectDataFromTableAndValidate();
            var issueCollection = this.getIssueCollection();
            var issues = issueCollection.getFieldNameAndContentAsArray();

            var errors = issueCollection.validate();
            if (errors.length) {
                this.renderErrorMessages(errors);
                return;
            }

            this.formObjectView.createIssues(issues);
        },

        submitFormWhenInvalid: function(errors) {
            this.renderErrorMessages(errors);
        },

        _bindOnSelectChangeEvent: function() {
            var self = this;

            _.each(this.mappingFields.getAllFieldNames(), function(mappingFieldName) {
                var fieldIdSelector = '#issue-' + mappingFieldName + '-index';
                var $select = $(fieldIdSelector);

                if ($select.length === 0) {
                    return;
                }

                // Apply AUI Selector2
                var $auiSelect2 = $select.auiSelect2().data('mappingFieldName', mappingFieldName);

                $select.off('change').on('change', self._onMappingFieldChanged.bind(self));
            });
        },

        _onMappingFieldChanged: function(e) {
            var $select = $(e.target);
            var fieldName = $select.data('mappingFieldName');

            // update index column for the changed field.
            var columnIndex = window.parseInt($select.val());
            if (!isNaN(columnIndex)) {
                var fieldModel = this.mappingFields.getIssueFieldByName(fieldName);
                fieldModel.set('index', columnIndex);
            }

            this.fillPreviewIssuesFromTable();
        },
    }, {
        MAX_ISSUE_PREVIEW: 3
    });

    return CreateMultipleIssuesView;
});
