;(function ($) {
    "use strict";

    /**
     * Create a jQuery plugin that supports sorting in server-side
     * Assure that table element do not have "aui-table-sortable" class
     * so that TableSorter plugin of AUI do not affect our expectation
     * Therefore, this jQuery plugin depends on "AJS.tablessortable" of AUI.
     *
     * @Usage: $("table).tablesorterServerOnly(options);
     * Custom options are described in below defaults options
     */
    $.extend({ tablesorterServerOnly: new function () {

        this.defaults = {
            //css class of header table to indicate disable sort
            classNameDisableSorting: "aui-table-column-unsortable",

            //css class of header table to indicate descending sort
            classNameHeaderDesc: "tablesorter-headerDesc",

            //css class of header table to indicate ascending sort
            classNameHeaderAsc: "tablesorter-headerAsc",

            // reverseSort is true that means descending sort (sort-up)
            // or false that means ascending sort (sort-down)
            reverseSort: false,

            //initial sort column name
            //column name is value of data-column-name of "table > thead > th"
            sortColumn: "",

            //callback function when initializing tablesorterServerOnly
            onInit: function (/*p*/) {},

            //callback function whenever sorting starts
            onSort: function (/*p*/) {},

            // stuff not set by the user
            events: {
                clickHeader: "click.sortServerOnly",
                refreshHeader: "refreshHeader.sortServerOnly",
                simulateClickHeader: "simulateClickHeader.sortServerOnly"
            }
        };

        var $this = this;

        var Util = {
            /**
             * Update headers of table to reflect current sort columns and status of sorting
             * @param table
             * @param p
             */
            updateCurrentHeaderSort: function (table, p) {
                Util.resetHeadersSort(table, p);
                p.$headers.each(function () {
                    var $this = $(this),
                        currentColumnName = $this.attr("data-column-name");

                    var isSortUp = p.reverseSort;
                    if (currentColumnName === p.sortColumn) {
                        isSortUp
                            ? $this.addClass(p.classNameHeaderDesc)
                            : $this.addClass(p.classNameHeaderAsc);
                    }
                });
            },

            /**
             * Update headers of table to reflect no sort state
             * @param table
             * @param p
             */
            resetHeadersSort: function (table, p) {
                p.$headers
                    .removeClass(p.classNameHeaderDesc)
                    .removeClass(p.classNameHeaderAsc);
            },

            prepareHTMLHeader: function (table, p) {
                p.$headers.not("." + p.classNameDisableSorting)
                    .attr('unselectable', 'on')
                    .bind('selectstart', false)
                    .addClass("tablesorter-header")
                    .wrapInner("<span class='aui-table-header-content'/>");
            },

            bindEvents: function (table , p) {
                var $table = $(table);

                $table.on(p.events.refreshHeader, function (/*e*/) {
                    Util.updateCurrentHeaderSort(table, p);
                });

                //for testable
                $table.on(p.events.simulateClickHeader, function (e, sortColumn, sortDirection) {
                    p.reverseSort = sortDirection;
                    p.sortColumn = sortColumn;
                });
            }
        };

        var init = function (table, p) {
            var $table = $(table);

            p.$table = $table;
            p.$headers = $table.find("thead th");
            p.$tbodies = $table.find("tbody");

            //init style of table header to indicate initial status of sort
            Util.prepareHTMLHeader(table, p);
            Util.updateCurrentHeaderSort(table, p);

            if (typeof p.onInit === "function") {
                p.onInit.apply(table, [p]);
            }

            //bind new click event on header to sort in server side
            p.$headers.on(p.events.clickHeader, function (/*e*/) {
                var $this = $(this);

                if ($this.hasClass(p.classNameDisableSorting)) {
                    return false;
                }

                //update current direction sort
                var newColumnName = $this.attr("data-column-name");
                if (newColumnName === p.sortColumn) {
                    p.reverseSort = !p.reverseSort;
                }
                else {
                    p.reverseSort = false;
                }

                p.sortColumn = newColumnName;

                if (typeof p.onSort === "function") {
                    p.onSort.apply(table, [p]);
                }

                return false;
            });

            Util.bindEvents(table, p);
        };

        $this.construct = function (settings) {
            return this.each(function () {
                var table = this,
                    p = $.extend( true, {}, $.tablesorterServerOnly.defaults, settings );

                // check if tablesorter has initialized then we destroy it
                if (this.config && table.hasInitialized && $.tablesorter) {
                    $.tablesorter.destroy(table, false, function () {
                        init(table, p);
                    });
                }
                else {
                    init(table, p);
                }
            });
        };

    }() });

    // extend plugin scope
    $.fn.extend({
        tablesorterServerOnly: $.tablesorterServerOnly.construct
    });

})(AJS.$);