define('confluence/ic/model/comment-collection',
    [
        'jquery',
        'backbone',
        'ajs',
        'underscore',
        'confluence/ic/model/comment'
    ], function(
        $,
        Backbone,
        AJS,
        _,
        Comment
    ) {

    "use strict";

    var CommentCollection = Backbone.Collection.extend({
        model: Comment,

        initialize: function() {
            this.listenTo(this, "sort", this._removeCachedMarkers);
        },

        url: function() {
            var contextPath = AJS.contextPath();
            var containerId = AJS.Meta.get('page-id');
            // Do we really need the containerType here? Can we infer it from the containerId
            return contextPath + "/rest/inlinecomments/1.0/comments?" +
                "containerId=" + containerId;
        },

        /**
         * Return number of resolved comment
         * @returns number
         */
        getResolvedCount: function() {
            return this.getResolvedComments().length;
        },

        /**
         * Return number of unresolve comment
         * @returns number
         */
        getUnresolveCount: function() {
            return this.reject(function(comment) {
                return comment.isResolved() === true && comment.isDeleted() === false;
            }).length;
        },

        getResolvedComments: function() {
            return this.filter(function(comment) {
                return comment.isResolved() === true && comment.isDeleted() === false;
            });
        },

        getResolvedCommentsDesc: function() {
            return  _.sortBy(this.getResolvedComments(), function(comment) {
                return 0 - comment.get("resolveProperties").resolvedTime;
            });
        },

        /**
         * Retrieves the next comment on the page from the collection. The currently displayed comment has the
         * active attribute set. The next comment is a comment following this comment in the sorted collection that
         * is not resolved and not dangling (a comment whose highlight marker has been deleted)
         * The next call does not wrap the collection and instead returns null if no appropriate comment exists after
         * the active comment
         * Also returns null if there is no active (currently displayed) comment in the collection
         * @returns the next comment model or null if no appropriate comment found
         */
        getNextCommentOnPage: function() {
            return this._getCommentAtRelativeOffset(1);
        },

        /**
         * Retrieves the previous comment on the page from the collection. The currently displayed comment has the
         * active attribute set. The previous comment is a comment preceding this comment in the sorted collection that
         * is not resolved and not dangling (a comment whose highlight marker has been deleted)
         * The previous call does not wrap the collection and instead returns null if no appropriate comment exists before
         * the active comment
         * Also returns null if there is no active (currently displayed) comment in the collection
         * @returns the previous comment model or null if no appropriate comment found
         */
        getPrevCommentOnPage: function() {
            return this._getCommentAtRelativeOffset(-1);
        },

        _getCommentAtRelativeOffset: function(offset) {
            var commentsOnPage = this.getCommentsOnPage();
            var activeComment = this.findWhere({ active: true });
            if (activeComment === undefined || commentsOnPage.length <= 1) {
                return null;
            }
            var commentsOnPageIds = _.pluck(commentsOnPage, "id");
            var index = _.indexOf(commentsOnPageIds, activeComment.get("id"));
            return commentsOnPage[(index + offset + commentsOnPage.length) % commentsOnPage.length];
        },

        getCommentsOnPage: function() {
            return this.filter(function(comment) {
                return ((comment.isResolved() === false) && !comment.isDangling() && (comment.isDeleted() === false)) || comment.get("active") === true;
            });
        },

        getCommentsOnPageCount: function() {
            return this.getCommentsOnPage().length;
        },

        getActiveIndexWithinPageComments: function() {
            var validCommentIds = _.pluck(this.getCommentsOnPage(), 'id');
            var activeComment = this.findWhere({ active: true });
            if (activeComment === undefined) {
                return null;
            }
            var activeCommentId = activeComment.get("id");
            return _.indexOf(validCommentIds, activeCommentId);
        },

        /*
         * The collection is always sorted by the markers on the page. Comments without markers are randomly interspersed
         * between the sorted markers. They are ignored when navigating to next/previous comment
         */
        comparator: function(modela, modelb) {
            // cache the markers as a property of the collection so we only do this lookup once per sort
            // this cache will be removed once the sort has finished
            if (this.markers === undefined) {
                this.markers = $('#content .wiki-content:first').find('.inline-comment-marker');
            }
            if (modela.highlight === undefined) {
                return 1;
            }
            if (modelb.highlight === undefined) {
                return -1;
            }
            return this.markers.index(modela.highlight) - this.markers.index(modelb.highlight);
        },

        // We want to look up the markers in the page only once per sort
        // In the comparator, we cache the looked up markers and at the start of a sort and use them throughout the
        // sorting process. After the sort is finished, we discard the cache and will create a new one next time we sort
        _removeCachedMarkers: function() {
            this.markers = undefined;
        }

    });
    
    return CommentCollection;
});