define('confluence/ic/view/highlight-text',
    [
        'backbone',
        'ajs',
        'confluence/ic/util/text-highlighter',
        'confluence/ic/util/utils'
    ],
    function(
        Backbone,
        AJS,
        TextHighlighter,
        Utils
    ) {

    "use strict";

    /*
     * Represents text highlighted on the page. Highlights are represented as
     * <span class="inline-comment-marker" data-ref="[some uuid]">text</span>
     * A highlight is marked as valid if it has a corresponding inline comment. It is possible to have highlights in
     * the DOM that have no corresponding comment. If a comment has a "active" attribute set to true, it is currently
     * focused on the page
     *
     * Also triggers a "comment:view" event when the user clicks on a "valid" text highlight
     */
    var HighlightTextView = Backbone.View.extend({
        el: "#content .wiki-content:first",

        events: {
            "click .inline-comment-marker.valid": "commentClick"
        },

        initialize: function () {
            if (this.collection) {
                // Research topic: what events do we need to listen for here? What even will the initial fetch
                // from the server trigger?
                this.listenTo(this.collection, 'sync remove change:deleted', this.render);
                this.listenTo(this.collection, 'change:active', this.setActive);
            }
            // When showing a comment in the sidebar, we don't want any temporary selections on the page
            this.listenTo(Backbone, "ic:view ic:overlap ic:sidebar-hidden", this.clearSelection);
            this.listenTo(Backbone, "ic:persist", this.persistComment);
            this.listenTo(Backbone, "ic:delete", this.render);

            AJS.bind("qr:add-new-highlight-text", this._addQuickReloadMarker);
        },

        // Remove any valid and active state from comment markers in the page, then iterate through all the
        // inline comments and mark the associated markers as valid. Run this anytime the list of inline comments changes.
        render: function () {
            this.inlineLinks = [];
            this.$('.inline-comment-marker').removeClass("valid");
            this.collection.each(function (comment) {
                if (!comment.isResolved() && !comment.get('deleted') && comment.highlights) {
                    comment.highlights.addClass('valid');
                    this._pushToInlineLinks(comment.highlights);
                }
            }, this);

            this._setupLinkDialog();
        },

        _pushToInlineLinks: function(highlights) {
            var thiz = this;
            highlights.each(function() {
                var $links  = Utils.getInlineLinks(this);
                $links.each(function() {
                    if($(this).attr('href')) {
                        thiz.inlineLinks.push($(this));
                    }
                });
            });
        },

        _setupLinkDialog: function() {

            var dialogOption = {
                width: 200,
                onHover: true,
                noBind: true,
                calculatePositions: this._calculatePositions,
                hideDelay: 1000
            };

            $(this.inlineLinks).each(function() {
                var link = this;
                AJS.InlineDialog(
                    link,
                    "inline-comment-link",
                    function($content, trigger, showPopup) {
                        $content.addClass('inline-comment-link');
                        $content.html(Confluence.Templates.IC.highlightTextLink({link: link.attr('href')}));
                        showPopup();
                        return false;
                    },
                    dialogOption
                );
            });
        },

        _calculatePositions: function(popup, targetPosition, mousePosition) {
            return {
                displayAbove: true,
                popupCss: {
                    top: targetPosition.target.offset().top - popup.height() - 10, // 10 = arrow height + padding 2px
                    left: mousePosition.x - (popup.width()/2)
                },
                arrowCss: {
                    top: popup.height(),
                    left: popup.width()/2 - 4 // 4 = arrow width / 2
                }
            };
        },

        // This sets the "blue" active highlight on the page. When a comment is being displayed, it's active attribute
        // is set to true. This causes the highlight to get and .active class. When a comment is closed, the active
        // attribute is removed.
        setActive: function(comment) {
            this.$('.inline-comment-marker').removeClass('active');
            var $marker = this.$('.inline-comment-marker[data-ref="' + comment.get('markerRef') + '"]');
            if (comment.get("active") && $marker.length) {
                $marker.addClass("active");
            }
        },

        commentClick: function (e) {
            e.preventDefault();
            var $highlight = this.$(e.currentTarget);
            if (!$highlight.hasClass("active")) {
                this.displayComment($highlight);
            } else {
                Backbone.trigger("ic:hide-sidebar", e, true);
            }
        },

        displayComment: function ($highlight, source) {
            var ref = $highlight.data('ref');
            var model = this.collection.findWhere({ markerRef: ref });
            Backbone.trigger("ic:view", model, source);
        },

        clearSelection: function () {
            (new TextHighlighter()).removeHighlight();
        },

        persistComment: function(comment) {
            this.$('.ic-current-selection').removeClass('ic-current-selection')
                .addClass("inline-comment-marker").attr('data-ref', comment.get("markerRef"));
            this.collection.add(comment);
        },

        _addQuickReloadMarker: function(e, commentAndUser, deferred) {
            var comment = commentAndUser.comment;
            var $marker = this.$('.inline-comment-marker[data-ref="' + comment.dataRef + '"]');
            if ($marker.length) { // check if the marker already exists
                deferred.resolve(commentAndUser);
                return;
            }
            var highlighted = (new TextHighlighter()).deserializeHighlights(comment.serializedHighlights, comment.dataRef);
            if (highlighted) {
                // if highlight success
                deferred.resolve(commentAndUser);
            } else {
                // if highlight failure
                deferred.reject(commentAndUser);
            }
        }
    });

    return HighlightTextView;
});
