define('confluence/ic/view/display-reply',
    [
        'jquery',
        'ajs',
        'underscore',
        'backbone',
        'confluence/ic/util/utils',
        'confluence/ic/view/create-comment',
        'confluence/ic/view/likes'
    ],
    function(
        $,
        AJS,
        _,
        Backbone,
        Utils,
        CreateCommentView,
        LikesView
    ) {

    "use strict";

    var DisplayReplyView = Backbone.View.extend({
        tagName: "div",

        className: "ic-display-reply",

        template: Confluence.Templates.IC.renderReply,

        events: {
            // also need to bind an event for canceling and saving the comment
            "click .ic-action-edit-reply": "editReply",
            "click .ic-action-delete-reply": "confirmDelete"
        },

        initialize: function() {
            _.bindAll(this, "onEditCallback");
            this.$el.attr('data-comment-id', this.model.get("id"));
        },

        render: function() {
            var data = _.extend(this.model.toJSON(), {
                resolved: this.options.resolved,
                darkFeatures: Utils.getDarkFeatures()
            });
            var html = this.template(data);
            this.$el.html(html);
            if (this.likesView) {
                this.likesView.remove();
            }
            this.likesView = new LikesView({contentId: this.model.id}).render();
            if(this.likesView.$el.is(':empty')) {
                this.$('.ic-action-like-reply').remove();
            } else {
                this.$('.ic-action-like-reply').html(this.likesView.el);
            }
            return this;
        },

        editReply: function(e) {
            var that = this;
            if (Utils.confirmProcess()) {
                var promise = Utils.getEditorFormat(this.model.get("id"));
                promise.done(function(editBody) {
                    that.editView = new CreateCommentView({
                        model: that.model,
                        type: "edit",
                        selection: {},
                        onFinished: that.onEditCallback,
                        commentText: editBody
                    });
                    that.$el.html(that.editView.$el);
                    that.editView.render();
                }).fail(function() {
                    Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.server.error.edit.reply'));
                });
            } else {
                // prevent Edit button has a green outline after cancel confirm dialog
                e && e.currentTarget.blur();
            }
        },

        onEditCallback: function() {
            this.editView && this.editView.destroy();
            this.editView = null;
            this.render();
        },

        deleteFail: function(model, response, options) {
            if (response.status === 401) {
                Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.delete.permission'));
            } else {
                Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.server.error'));
            }
        },

        confirmDelete: function(e) {
            var answer = window.confirm(AJS.I18n.getText('inline.comments.delete.reply.confirmation.message'));
            if (answer === true) {
                this.deleteReply(e);
            }
        },

        deleteReply: function(e) {
            // disable the delete button, so we can only click it once
            $(e.target).attr("aria-disabled", true).prop("disabled", true);
            this.model.destroy({
                wait: true,
                error: this.deleteFail,
                success: function() {
                    Backbone.trigger("ic:reply:delete");
                }
            });
        },

        destroy: function() {
            this.likesView.remove();
            this.editView && this.editView.destroy();
            this.remove();
        }
    });

    return DisplayReplyView;
});
