/**
 * An improved version of the inline dialog that scrolls correctly when using documentation theme
 * Because documentation theme uses a scrollable, fixed-height container, inline dialogs in documentation theme
 * would not scroll with the content. This fixes that issue
 *
 * Call this exactly as you would call AJS.InlineDialog()
 * https://developer.atlassian.com/display/AUI/Inline+Dialog
 *
 * @param items The elements that will trigger the inline-dialog, use a jQuery Selector to select items
 * @param identifier A unique identifier for the inline-dialog
 * @param url can be a url with dialog contents or a function to generate the dialog contents dynamically
 * @param options A number of different options may be passed in as an associative array
 * @returns the inline dialog
 */
Confluence.ScrollingInlineDialog = function(items, identifier, url, options) {
    // scrolling container in the doc theme
    var $docThemeContainer = Confluence.DocThemeUtils.getDocThemeContentElement();
    var isDocTheme = Confluence.DocThemeUtils.isDocTheme();
    options = options || {};

    // if using doctheme attach the inline dialog to the doc theme container, else attached to body by default
    if (!options.container && isDocTheme) {
        options.container = $docThemeContainer;
    }

    /**
     * This method is copied from Inline Dialog and MODIFIED to work with documentation theme
     *
     * Modifications:
     * Use position() instead of offset(), this returns the targetPosition coordinates relative to containing element
     *     and not the viewport
     * Adjust top values for scroll in docTheme
     * Use either window for the container or the docThemeContainer
     * has an extra isActionPanelDialog parameter to help position that dialog accordingly
     */
    var calculatePositions = function (popup, targetPosition, mousePosition, opts) {
        var popupLeft;    //position of the left edge of popup box from the left of the screen
        var popupRight = "auto";   //position of the right edge of the popup box fron the right edge of the screen
        var popupTop;   //position of the top edge of popup box
        var arrowOffsetY = -7;    //the offsets of the arrow from the top edge of the popup, default is the height of the arrow above the popup
        var arrowOffsetX;
        var displayAbove;   //determines if popup should be displayed above the the trigger or not
        var containerWidth = isDocTheme ? $docThemeContainer.width() : AJS.$(window).width();

        var targetOffset = targetPosition.target.position();
        targetOffset.top += isDocTheme ? $docThemeContainer.scrollTop() : 0;
        targetOffset.left += isDocTheme ? $docThemeContainer.scrollLeft() : 0;

        var triggerWidth = targetPosition.target.outerWidth(); //The total width of the trigger (including padding)
        var middleOfTrigger = targetOffset.left + triggerWidth / 2;    //The absolute x position of the middle of the Trigger

        var bottomOfViewablePage = isDocTheme ?
                                        $docThemeContainer.scrollTop() + AJS.$(window).height() - AJS.$('#footer').outerHeight() :
                                        (window.pageYOffset || document.documentElement.scrollTop) + AJS.$(window).height();

        //CONSTANTS
        var SCREEN_PADDING = 10; //determines how close to the edge the dialog needs to be before it is considered offscreen
        var SCREEN_TOP_PADDING = 20; //keep the bottom of inline-dialog enough far from the bottom of browser.

        popupTop = targetOffset.top + targetPosition.target.outerHeight() + opts.offsetY;
        var arrowWidth = popup.find(".arrow").outerWidth();
        var popupOuterWidth = popup.outerWidth();
        var targetOuterWidth = targetPosition.target.outerWidth();
        if (opts.centerOnHighlight) {
            // if action panel is wider that the selection, center the arrow and the dialog above the selection
            if (popupOuterWidth > targetOuterWidth) {
                popupLeft = targetOffset.left - (popupOuterWidth - targetOuterWidth) / 2;
                arrowOffsetX = middleOfTrigger - popupLeft - arrowWidth/2;
            } else {
                // align left to selection and place arrow in the center of dialog
                popupLeft = targetOffset.left + opts.offsetX;
                arrowOffsetX = (popupOuterWidth - arrowWidth) / 2;
            }
        } else {
            // if not action panel, always align left the dialog to the selection
            popupLeft = targetOffset.left + opts.offsetX;
            // if the create issue popup is wider than the selection, center the arrow on the selection
            if (popupOuterWidth > targetOuterWidth) {
                arrowOffsetX = middleOfTrigger - popupLeft - arrowWidth/2;
            } else {
                // if create issue popup is narrow then selection, place the arrow in the enter of the dialog
                arrowOffsetX = (popupOuterWidth - arrowWidth) / 2;
            }
        }
        // arrow always needs to have a positive offset
        arrowOffsetX = (arrowOffsetX < 0) ? 0 : arrowOffsetX;

        // Fix default behavior of original inline-dialog code.
        var distanceToTopOfViewablePage = (isDocTheme) ?
                                            (targetOffset.top - $docThemeContainer.scrollTop()) :
                                            (targetOffset.top - (window.pageYOffset || document.documentElement.scrollTop));

        var popupMaxHeight = opts.maxHeight || 0;
        var popupHeight = popup.height();
        var enoughRoomAbove = distanceToTopOfViewablePage > Math.max(popupHeight, popupMaxHeight);
        var enoughRoomBelow = (popupTop + popup.height()) < bottomOfViewablePage;
        //Check if the popup should be displayed above the trigger or not (if the user has set opts.onTop to true and if theres enough room)
        displayAbove =  (!enoughRoomBelow && enoughRoomAbove) || opts.onTop;

        // opts is a reference to the original options that we used at init config
        // Assign displayAbove value into onTop property to prevent dialog is flipped after call refresh()
        opts.onTop = displayAbove;

        //calculate if the popup will be offscreen
        var diff = containerWidth - (popupLeft  + popupOuterWidth + SCREEN_PADDING);

        //check if the popup should be displayed above or below the trigger
        if (displayAbove) {
            popupTop = targetOffset.top - popupHeight - 8; //calculate the flipped position of the popup (the 8 allows for room for the arrow
            arrowOffsetY = popupHeight;
        }

        // We have to make sure inline-dialog never clipped by the viewport
        if (displayAbove === false && enoughRoomBelow === false) {
            var clippedHeight = (popupTop + popupHeight) - bottomOfViewablePage;
            var optimalScrollTop = isDocTheme ?
                                    $docThemeContainer.scrollTop() + clippedHeight + AJS.$('#footer').outerHeight() :
                                    (window.pageYOffset || document.documentElement.scrollTop) + clippedHeight + SCREEN_TOP_PADDING;
            var $container = isDocTheme ? $docThemeContainer : AJS.$('html, body');
            $container.stop().animate({
                scrollTop: optimalScrollTop
            }, 500);
        }

        //check if the popup should show up relative to the mouse
        if (opts.isRelativeToMouse) {
            if (diff < 0) {
                popupRight = SCREEN_PADDING;
                popupLeft = "auto";
                arrowOffsetX = mousePosition.x - (AJS.$(window).width() - opts.width);
            } else {
                popupLeft = mousePosition.x - 20;
                arrowOffsetX = mousePosition.x - popupLeft;
            }
        } else {
            if (diff < 0) {
                popupRight = SCREEN_PADDING;
                popupLeft = "auto";

                var popupRightEdge = containerWidth - popupRight;
                var popupLeftEdge = popupRightEdge - popupOuterWidth;
                //arrow's position must be relative to the popup's position and not of the screen.
                arrowOffsetX = middleOfTrigger - popupLeftEdge - arrowWidth / 2;
            }
        }
        return {
            displayAbove: displayAbove,
            popupCss: {
                left: popupLeft,
                right: popupRight,
                top: popupTop
            },
            arrowCss: {
                position: "absolute",
                left : arrowOffsetX,
                right : "auto",
                top : arrowOffsetY
            }
        }
    };

    // still allow the user to override our custom calculatePositions if needed
    if (!options.calculatePositions) {
        options.calculatePositions = calculatePositions;
    }

    return AJS.InlineDialog.call(this, items, identifier, url, options);
};