package com.atlassian.confluence.plugins.createcontent.rest;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.pages.templates.PageTemplateManager;
import com.atlassian.confluence.plugins.createcontent.ContentTemplateRefManager;
import com.atlassian.confluence.plugins.createcontent.SpaceBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.activeobjects.ContentTemplateRefAo;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.BlueprintIllegalArgumentException;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.ResourceErrorType;
import com.atlassian.confluence.plugins.createcontent.api.services.SpaceBlueprintService;
import com.atlassian.confluence.plugins.createcontent.exceptions.ResourceException;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.confluence.plugins.createcontent.impl.SpaceBlueprint;
import com.atlassian.confluence.plugins.createcontent.rest.entities.BlueprintSpaceEntity;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreateBlueprintSpaceRestEntity;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreatePersonalSpaceRestEntity;
import com.atlassian.confluence.plugins.createcontent.services.model.BlueprintSpace;
import com.atlassian.confluence.plugins.createcontent.template.PluginPageTemplateHelper;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.security.SpacePermissionManager;
import com.atlassian.confluence.setup.settings.SettingsManager;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.sal.api.transaction.TransactionCallback;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.core.Response;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static javax.ws.rs.core.MediaType.APPLICATION_XML;

/**
 * For retrieving Space Blueprints and creating Spaces from them.
 *
 * @since 1.6
 */
@Path("/space-blueprint")
public class SpaceBlueprintResource extends AbstractRestResource {
    public static final String PARAM_KEY = "key";
    public static final String PARAM_ID = "id";

    private final SpaceBlueprintService spaceBlueprintService;
    private final SpaceBlueprintManager spaceBlueprintManager;
    private final ContentTemplateRefManager contentTemplateRefManager;
    private final PluginPageTemplateHelper pageTemplateHelper;
    private final PageTemplateManager pageTemplateManager;
    private final ActiveObjects activeObjects;
    private final SettingsManager settingsManager;

    public SpaceBlueprintResource(final SpaceBlueprintService spaceBlueprintService, final PermissionManager permissionManager,
                                  SpaceManager spaceManager, SpacePermissionManager spacePermissionManager,
                                  @Qualifier("spaceBlueprintManager") final SpaceBlueprintManager spaceBlueprintManager,
                                  final ContentTemplateRefManager contentTemplateRefManager,
                                  final PluginPageTemplateHelper pageTemplateHelper, final PageTemplateManager pageTemplateManager,
                                  final ActiveObjects activeObjects, final SettingsManager settingsManager) {
        super(permissionManager, spaceManager, spacePermissionManager);

        this.spaceBlueprintService = spaceBlueprintService;
        this.spaceBlueprintManager = spaceBlueprintManager;
        this.contentTemplateRefManager = contentTemplateRefManager;
        this.pageTemplateHelper = pageTemplateHelper;
        this.pageTemplateManager = pageTemplateManager;
        this.activeObjects = activeObjects;
        this.settingsManager = settingsManager;
    }

    /**
     * Create a space from a given Space Blueprint.
     */
    @POST
    @Path("create-space")
    @Consumes({APPLICATION_JSON, APPLICATION_XML})
    public BlueprintSpaceEntity createSpace(CreateBlueprintSpaceRestEntity entity) throws BlueprintIllegalArgumentException {
        BlueprintSpace space = spaceBlueprintService.createSpace(entity, getUser());
        String baseUrl = settingsManager.getGlobalSettings().getBaseUrl();
        return new BlueprintSpaceEntity(space, baseUrl);
    }

    /**
     * Create a space from a given Space Blueprint.
     */
    @POST
    @Path("create-personal-space")
    @Consumes({APPLICATION_JSON, APPLICATION_XML})
    public BlueprintSpaceEntity createPersonalSpace(CreatePersonalSpaceRestEntity entity) {
        checkNullEntity(entity);

        BlueprintSpace space = spaceBlueprintService.createPersonalSpace(entity, getUser());
        String baseUrl = settingsManager.getGlobalSettings().getBaseUrl();
        return new BlueprintSpaceEntity(space, baseUrl);
    }

    // FIXME - make path more RESTy
    @GET
    @Path("byKey/{" + PARAM_KEY + "}")
    public SpaceBlueprint getByModuleCompleteKey(@PathParam(PARAM_KEY) String moduleCompleteKey) {
        checkAdminPermission();
        checkEmptyParameter(moduleCompleteKey, PARAM_KEY);

        return spaceBlueprintManager.getCloneByModuleCompleteKey(new ModuleCompleteKey(moduleCompleteKey));
    }

    // TODO - shouldn't need the "get" part of the path.
    @GET
    @Path("get/{" + PARAM_ID + "}")
    public SpaceBlueprint get(@PathParam(PARAM_ID) UUID id) {
        checkAdminPermission();
        checkNullParameter(id, PARAM_ID);

        return spaceBlueprintManager.getById(id);
    }

    @GET
    @Path("list")
    public List<SpaceBlueprint> getAllSpaceBlueprints() {
        checkAdminPermission();

        return spaceBlueprintManager.getAll();
    }

    @PUT
    @Path("save")
    @Consumes({APPLICATION_JSON, APPLICATION_XML})
    public SpaceBlueprint save(SpaceBlueprint blueprint) {
        checkAdminPermission();

        return spaceBlueprintManager.update(blueprint);
    }

    @POST
    @Path("create")
    @Consumes({APPLICATION_JSON, APPLICATION_XML})
    public SpaceBlueprint create(@Nonnull SpaceBlueprintEntity entity) {
        checkAdminPermission();

        // Home page
        final UUID newHomePageId;
        final ContentTemplateRefAo newHomePage;

        final UUID homePageId = entity.getHomePageId();
        final Long homePageTemplateId = entity.getHomePageTemplateId();

        if (homePageId != null) {
            // Create a copy of this homepage
            ContentTemplateRef contentTemplateRef = contentTemplateRefManager.getById(homePageId);
            if (contentTemplateRef == null) {
                throw new ResourceException("The specified homePageId was not found", Response.Status.NOT_FOUND, ResourceErrorType.NOT_FOUND_CONTENT_TEMPLATE_REF, homePageId);
            }

            newHomePage = createContentTemplateRefAo(contentTemplateRef);
        } else if (homePageTemplateId != null && homePageTemplateId > 0) {
            // FIXME: Pass a name here
            newHomePage = createContentTemplateRefAo(new ContentTemplateRef(null, homePageTemplateId, null, null, false, null));
        } else {
            newHomePage = null;
        }
        newHomePageId = newHomePage != null
                ? UUID.fromString(newHomePage.getUuid())
                : null;

        return spaceBlueprintManager.create(entity.getSpaceBlueprint(), newHomePageId);
    }

    private ContentTemplateRefAo createContentTemplateRefAo(@Nonnull final ContentTemplateRef contentTemplateRef) {
        // FIXME: We shouldn't be using an ActiveObjects object here, but a manager, even if it's just for doing save()
        return activeObjects.executeInTransaction(new TransactionCallback<ContentTemplateRefAo>() {
            @Override
            public ContentTemplateRefAo doInTransaction() {
                return doCreateContentTemplateRefAo(contentTemplateRef, null);
            }
        });
    }

    @Nonnull
    private ContentTemplateRefAo doCreateContentTemplateRefAo(@Nonnull final ContentTemplateRef contentTemplateRef, @Nullable final ContentTemplateRefAo parent) {
        PageTemplate pageTemplate = pageTemplateHelper.getPageTemplate(contentTemplateRef);
        pageTemplate.setId(0);
        // Create a clone
        pageTemplateManager.savePageTemplate(pageTemplate, null);

        // Create a reference
        List<ContentTemplateRef> children = new ArrayList<ContentTemplateRef>(contentTemplateRef.getChildren());
        contentTemplateRef.getChildren().clear();
        ContentTemplateRefAo result = contentTemplateRefManager.createAo(contentTemplateRef);
        // Point the reference to the new page template
        result.setParent(parent);
        result.setTemplateId(pageTemplate.getId());
        result.save();

        // Do the same for the children
        for (ContentTemplateRef child : children) {
            doCreateContentTemplateRefAo(child, result);
        }

        return result;
    }

    @DELETE
    @Path("deleteAll")
    public int deleteAll() {
        checkAdminPermission();

        return spaceBlueprintManager.deleteAll();
    }
}
