package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.core.DefaultDeleteContext;
import com.atlassian.confluence.core.DefaultSaveContext;
import com.atlassian.confluence.core.SaveContext;
import com.atlassian.confluence.mail.notification.NotificationManager;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.pages.PageManager;
import com.atlassian.confluence.pages.PageUpdateTrigger;
import com.atlassian.confluence.pages.TrashManager;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.ContentTemplateRefManager;
import com.atlassian.confluence.plugins.createcontent.actions.BlueprintManager;
import com.atlassian.confluence.plugins.createcontent.api.events.SpaceBlueprintCreateEvent;
import com.atlassian.confluence.plugins.createcontent.api.events.SpaceBlueprintHomePageCreateEvent;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.BlueprintIllegalArgumentException;
import com.atlassian.confluence.plugins.createcontent.api.services.SpaceBlueprintService;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreatePersonalSpaceRestEntity;
import com.atlassian.confluence.plugins.createcontent.services.PromotedBlueprintService;
import com.atlassian.confluence.plugins.createcontent.services.RequestResolver;
import com.atlassian.confluence.plugins.createcontent.services.model.BlueprintSpace;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintSpaceEntity;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintSpaceRequest;
import com.atlassian.confluence.plugins.createcontent.services.model.CreatePersonalSpaceRequest;
import com.atlassian.confluence.search.lucene.ConfluenceIndexManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.ModuleCompleteKey;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import static com.google.common.collect.Lists.newArrayList;

/**
 * @since 1.6
 */
public class DefaultSpaceBlueprintService implements SpaceBlueprintService {
    private static final String PRIVATE = "private";

    private final SpaceManager spaceManager;
    private final PageManager pageManager;
    private final TrashManager trashManager;
    private final ContentTemplateRefManager contentTemplateRefManager;
    private final BlueprintManager blueprintManager;
    private final EventPublisher eventPublisher;
    private final ContentBlueprintManager contentBlueprintManager;
    private final RequestResolver requestResolver;
    private final NotificationManager notificationManager;
    private final ConfluenceIndexManager indexManager;
    private final PromotedBlueprintService promotedBlueprintService;
    private org.slf4j.Logger log = LoggerFactory.getLogger(DefaultSpaceBlueprintService.class);

    public DefaultSpaceBlueprintService(SpaceManager spaceManager, PageManager pageManager,
                                        TrashManager trashManager, ContentTemplateRefManager contentTemplateRefManager,
                                        BlueprintManager blueprintManager,
                                        final EventPublisher eventPublisher, final ContentBlueprintManager contentBlueprintManager,
                                        RequestResolver requestResolver, NotificationManager notificationManager,
                                        ConfluenceIndexManager indexManager, PromotedBlueprintService promotedBlueprintService) {
        this.spaceManager = spaceManager;
        this.pageManager = pageManager;
        this.trashManager = trashManager;
        this.contentTemplateRefManager = contentTemplateRefManager;
        this.blueprintManager = blueprintManager;
        this.eventPublisher = eventPublisher;
        this.contentBlueprintManager = contentBlueprintManager;
        this.requestResolver = requestResolver;
        this.notificationManager = notificationManager;
        this.indexManager = indexManager;
        this.promotedBlueprintService = promotedBlueprintService;
    }

    @Override
    public BlueprintSpace createSpace(@Nonnull CreateBlueprintSpaceEntity entity, @Nullable ConfluenceUser creator) throws
            BlueprintIllegalArgumentException {
        CreateBlueprintSpaceRequest createRequest = requestResolver.resolve(entity, creator);

        String key = createRequest.getSpaceKey();
        String name = createRequest.getName();
        String description = createRequest.getDescription();
        SpaceBlueprint spaceBlueprint = createRequest.getBlueprint();
        Map<String, Object> context = createRequest.getContext();

        Space space;
        if (PRIVATE.equalsIgnoreCase(entity.getPermissions()))
            space = spaceManager.createPrivateSpace(key, name, description, creator);
        else
            space = spaceManager.createSpace(key, name, description, creator);

        eventPublisher.publish(new SpaceBlueprintCreateEvent(this, space, spaceBlueprint, creator, context));

        if (setHomePage(createRequest, creator, space)) {
            eventPublisher.publish(new SpaceBlueprintHomePageCreateEvent(this, space, spaceBlueprint, creator, context));
        }

        addPromotedBps(spaceBlueprint, space, creator);

        return new BlueprintSpace(space);
    }

    @Override
    public BlueprintSpace createPersonalSpace(CreatePersonalSpaceRestEntity entity, ConfluenceUser creator) {
        CreatePersonalSpaceRequest request = requestResolver.resolve(entity, creator);
        ConfluenceUser spaceUser = request.getSpaceUser();
        String spaceUserFullName = spaceUser.getFullName();

        // FIXME - this code, and the code in CreatePersonalSpaceAction and SpacesSoapService should all
        // delegate to a single service.
        Space space;
        if (PRIVATE.equals(request.getSpacePermission())) {
            space = spaceManager.createPrivatePersonalSpace(spaceUserFullName, null, creator);
        } else {
            space = spaceManager.createPersonalSpace(spaceUserFullName, null, creator);
        }

        // Users watch their own spaces by default
        notificationManager.addSpaceNotification(spaceUser, space);

        // Make sure the space shows up on the front page immediately
        indexManager.flushQueue(ConfluenceIndexManager.IndexQueueFlushMode.ONLY_FIRST_BATCH);

        return new BlueprintSpace(space);
    }

    private boolean setHomePage(CreateBlueprintSpaceRequest createRequest, ConfluenceUser creator, Space space) {
        UUID homePageId = createRequest.getBlueprint().getHomePageId();
        if (homePageId == null)
            return false;

        /*
            Due to the way that the Home page of the space currently gets created (via a create-event handler called
            InitialSpaceContentListener), we let the "normal" home page get created and then replace it with our
            Blueprint-sourced one.

            The alternative would be to add an unpersisted home page to the Space being created and update
            InitialSpaceContentListener to not do anything if this home page is detected. Then, we would rely on
            the Hibernate cascade to save the page when the space is saved, and hope that all of the right logic got
            called.

            That's not all - in addition to this cascade magic we have the issue of Blueprints having the ability to
            fire custom events, save labels and permissions and much, much more. These abilities would need detaching
            from the Blueprint page save (because the Space save would do it) and calling _after_ the openCreateSpaceDialog line.

            The current logic seems less convoluted, even if it has the inefficiency of a dummy Home page being created,
            only to be blown away. If you've read this far, feel free to come up with a better solution :) dT
         */

        Page oldHomePage = space.getHomePage();

        ContentTemplateRef contentTemplateRef = contentTemplateRefManager.getById(homePageId);
        SaveContext saveContext = DefaultSaveContext.builder()
                .updateLastModifier(true)
                .updateTrigger(PageUpdateTrigger.SPACE_CREATE)
                .build();
        final Page newHomePage = blueprintManager.createPageFromTemplate(contentTemplateRef, creator, space, null,
                createRequest.getContext(), saveContext);

        space.setHomePage(newHomePage);
        spaceManager.saveSpace(space);
        pageManager.trashPage(oldHomePage, DefaultDeleteContext.DEFAULT);   // awwwwwwwww
        trashManager.purge(space.getKey(), oldHomePage.getId());

        return true;
    }

    private void addPromotedBps(final SpaceBlueprint spaceBlueprint, final Space space, ConfluenceUser creator) {
        // 1. Get promoted blueprints
        List<ModuleCompleteKey> promotedBps = spaceBlueprint.getPromotedBps();
        if (promotedBps == null || promotedBps.isEmpty()) {
            return;
        }

        // 2. Convert them to UUIDs
        final List<String> uuids = newArrayList();
        for (ModuleCompleteKey promotedBp : promotedBps) {
            ContentBlueprint pluginBlueprint = contentBlueprintManager.getPluginBlueprint(promotedBp);
            if (pluginBlueprint != null) {
                uuids.add(pluginBlueprint.getId().toString());
            } else {
                log.warn("Could not find plugin blueprint for blueprint with moduleCompleteKey " + promotedBp + " when creating space for user " + creator.getFullName());
            }
        }

        // 3. Store UUIDs against the space
        promotedBlueprintService.promoteBlueprints(uuids, space);
    }
}
