package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.plugins.createcontent.BlueprintConstants;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreateDialogWebItemEntity;
import com.atlassian.confluence.plugins.createcontent.services.BlueprintSorter;
import com.atlassian.confluence.plugins.createcontent.services.PromotedBlueprintService;
import com.atlassian.confluence.plugins.createcontent.services.PromotedTemplateService;
import com.atlassian.confluence.plugins.createcontent.template.PageTemplateGrouper;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.user.ConfluenceUser;
import com.google.common.collect.Lists;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.UUID;

import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.MODULE_KEY_BLANK_SPACE_ITEM;
import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.MODULE_KEY_PERSONAL_SPACE_ITEM;
import static com.google.common.collect.Lists.newArrayList;
import static java.util.Collections.sort;

public class DefaultBlueprintSorter implements BlueprintSorter {
    /**
     * Sort blueprint and template web items. Blueprints are always listed first.
     */
    private static final Comparator<CreateDialogWebItemEntity> WEB_ITEM_COMPARATOR = new Comparator<CreateDialogWebItemEntity>() {
        @Override
        public int compare(CreateDialogWebItemEntity o1, CreateDialogWebItemEntity o2) {
            boolean isO1Blueprint = o1.getContentBlueprintId() != null;
            boolean isO2Blueprint = o2.getContentBlueprintId() != null;
            if (isO1Blueprint && !isO2Blueprint)
                return -1;

            if (!isO1Blueprint && isO2Blueprint)
                return 1;

            return o1.getName().compareToIgnoreCase(o2.getName());
        }

    };

    private final PageTemplateGrouper pageTemplateGrouper;
    private final PromotedBlueprintService promotedBlueprintService;
    private final PromotedTemplateService promotedTemplateService;

    public DefaultBlueprintSorter(PageTemplateGrouper pageTemplateGrouper,
                                  PromotedBlueprintService promotedBlueprintService, PromotedTemplateService promotedTemplateService) {
        this.pageTemplateGrouper = pageTemplateGrouper;
        this.promotedBlueprintService = promotedBlueprintService;
        this.promotedTemplateService = promotedTemplateService;
    }

    @Override
    public List<CreateDialogWebItemEntity> sortContentBlueprintItems(@Nonnull Collection<CreateDialogWebItemEntity> items,
                                                                     @Nonnull Space space, @Nonnull ConfluenceUser user) {
        CreateDialogWebItemEntity blankPageItem = removeFromItems(items, BlueprintConstants.MODULE_KEY_BLANK_PAGE);
        CreateDialogWebItemEntity blogPostItem = removeFromItems(items, BlueprintConstants.MODULE_KEY_BLOG_POST);

        List<CreateDialogWebItemEntity> sortedItems = sortWebItems(items);
        if (blogPostItem != null) {
            // BAD performance! Maybe.
            sortedItems.add(0, blogPostItem);
        }
        if (blankPageItem != null) {
            sortedItems.add(0, blankPageItem);
        }

        return promoteItemsForSpace(space, sortedItems);
    }

    @Override
    public List<CreateDialogWebItemEntity> sortSpaceBlueprintItems(@Nonnull List<CreateDialogWebItemEntity> items,
                                                                   @Nonnull ConfluenceUser user) {
        final CreateDialogWebItemEntity blankSpaceItem = removeFromItems(items, MODULE_KEY_BLANK_SPACE_ITEM);
        final CreateDialogWebItemEntity personalSpaceItem = removeFromItems(items, MODULE_KEY_PERSONAL_SPACE_ITEM);

        List<CreateDialogWebItemEntity> sortedItems = sortWebItems(items);

        List<CreateDialogWebItemEntity> enabledSystemBPs = Lists.newArrayList();

        if (blankSpaceItem != null)
            enabledSystemBPs.add(blankSpaceItem);
        if (personalSpaceItem != null)
            enabledSystemBPs.add(personalSpaceItem);

        if (enabledSystemBPs.size() > 0) {
            sortedItems.addAll(0, enabledSystemBPs);  // not too worried about performance on this EPIC list. Yet.
        }

        return sortedItems;
    }

    private CreateDialogWebItemEntity removeFromItems(@Nonnull Collection<CreateDialogWebItemEntity> items, @Nonnull String itemModuleKey) {
        for (CreateDialogWebItemEntity item : items) {
            if (itemModuleKey.equals(item.getItemModuleCompleteKey())) {
                items.remove(item);
                return item;
            }
        }
        return null;
    }

    private List<CreateDialogWebItemEntity> sortWebItems(@Nonnull Collection<CreateDialogWebItemEntity> pluginItems) {
        final List<CreateDialogWebItemEntity> sortedItems = newArrayList(pluginItems);
        sort(sortedItems, WEB_ITEM_COMPARATOR);

        return sortedItems;
    }

    // If we have promoted blueprints, we should put them first
    private List<CreateDialogWebItemEntity> promoteItemsForSpace(@Nonnull Space space,
                                                                 @Nonnull List<CreateDialogWebItemEntity> webItemsEntities) {
        final Collection<ContentBlueprint> blueprintsInSpace = pageTemplateGrouper.getSpaceContentBlueprints(space);

        final Collection<ContentBlueprint> promotedBps = promotedBlueprintService.getPromotedBlueprints(blueprintsInSpace, space);
        final Collection<Long> promotedTemplates = promotedTemplateService.getPromotedTemplates(space);

        if (promotedBps.isEmpty() && promotedTemplates.isEmpty()) {
            return webItemsEntities;
        }

        final Collection<UUID> promotedBpsUuids = convertBlueprintsToUuid(promotedBps);

        for (int j = webItemsEntities.size() - 1; j >= 0; j--) {
            final CreateDialogWebItemEntity webItem = webItemsEntities.get(j);
            UUID contentBlueprintId = webItem.getContentBlueprintId();
            String templateId = webItem.getTemplateId();

            boolean isPromotedBlueprint = contentBlueprintId != null && promotedBpsUuids.contains(contentBlueprintId);
            boolean isPromotedTemplate = templateId != null && promotedTemplates.contains(Long.parseLong(templateId));

            if (!webItem.isPromoted() && (isPromotedBlueprint || isPromotedTemplate)) {
                webItem.setPromoted(true);
                webItemsEntities.remove(j++);    // We need not to skip anything, so we increment the index again
                webItemsEntities.add(0, webItem);
            }
        }
        return webItemsEntities;
    }

    private Collection<UUID> convertBlueprintsToUuid(@Nonnull Collection<ContentBlueprint> blueprints) {
        Collection<UUID> ids = Lists.newArrayList();

        for (ContentBlueprint blueprint : blueprints) {
            ids.add(blueprint.getId());
        }

        return ids;
    }
}
